﻿:Namespace  Files ⍝ V1.1
⍝ Various file ustilities

    (⎕IO ⎕ML)←1 0  ⋄  NUL←⎕UCS 0

    A←'A*'[1+80∊⎕DR'']{⍺,⍺⍺,⍵} ⍝ use A or * in ⎕NA statements

    ∇ r←GLE dummy;gle
    ⍝ Return last error number
      'gle'⎕NA'I4 kernel32.C32|GetLastError'
      r←gle  ⍝ return an error number
    ∇

    ∇ r←GetLastError msg
      r←GLE 0 ⋄ →(0∊⍴msg)⍴0  ⍝ return an error number
      r←msg,⍕r               ⍝ or a string message
    ∇

    ⍝ Signal last error + 100 with message
    Error←⊢(⊢ ⎕signal⍨ ⊣,(⍕⊢)) 100+GLE

    ∇ r←text AppendText name;tn
     ⍝ Append text to existing file (must be single byte text)
      tn←name ⎕NTIE 0
      r←text ⎕NAPPEND tn(⎕DR' ')
      ⎕NUNTIE tn
    ∇

    ∇ {protect}Copy FmTo;CopyFileX    ⍝ Copy file Fm -> To
      :If 900⌶⍬          ⍝ Copy fails if <protect> and 'To' exists.
          protect←0      ⍝ Default unprotected copy.
      :EndIf
      'CopyFileX'⎕NA'I kernel32.C32|CopyFile'A' <0T <0T I'
      :If 0=CopyFileX FmTo,protect
          Error'CopyFile error:'
      :EndIf
    ∇

    ∇ Delete name;DeleteFileX
      'DeleteFileX'⎕NA'I kernel32.C32|DeleteFile'A' <0T'
      :If 0=DeleteFileX⊂name
          Error'DeleteFile error:'
      :EndIf
    ∇

    ∇ rslt←{amsk}Dir path;handle;next;ok;⎕IO;attrs;FindFirstFileX;FindNextFileX;FindClose;FileTimeToLocalFileTime;FileTimeToSystemTime;GetLastError
     ⍝ Amsk is a 32 element bool attribute mask.
     ⍝ Only files with attributes corresponding to 1-s in the mask will be returned.
     ⍝ '*'s mark default attribute mask.
     ⍝
     ⍝        * [31] <=> READONLY
     ⍝          [30] <=> HIDDEN
     ⍝        * [29] <=> SYSTEM
     ⍝          [28] <=> undocumented
     ⍝        * [27] <=> DIRECTORY
     ⍝        * [26] <=> ARCHIVE
     ⍝          [25] <=> DEVICE
     ⍝        * [24] <=> NORMAL - only set if no other bits are set
     ⍝        * [23] <=> TEMPORARY
     ⍝        * [22] <=> SPARSE FILE
     ⍝        * [21] <=> REPARSE POINT
     ⍝        * [20] <=> COMPRESSED
     ⍝        * [19] <=> OFFLINE
     ⍝        * [18] <=> NOT CONTENT INDEXED
     ⍝        * [17] <=> ENCRYPTED
     ⍝        * rest <=> undocumented (but in the default set so that
     ⍝                   Microsoft can extend them)
     ⍝ rslt is a vector of character vectors of filenames
     
      ⎕IO←0
      :If 900⌶⍬ ⋄ amsk←~(⍳32)∊30 28 25 ⋄ :EndIf  ⍝ Default attribute mask.
      _FindDefine
      (handle rslt)←_FindFirstFile path
      ('file not found (ntdir error:',(⍕rslt),')')⎕SIGNAL handle↓102   ⍝ file not found
      rslt←,⊂rslt
      :While 1=0⊃(ok next)←_FindNextFile handle
          rslt,←⊂next
      :EndWhile
      :If 0 18∨.≠ok next
          ('ntdir error:',⍕next)⎕SIGNAL 11       ⍝ DOMAIN
      :EndIf
      ok←FindClose handle
      rslt←↓[0]↑rslt
      attrs←(32⍴2)⊤0⊃rslt                        ⍝ Get attributes into bits
      rslt←(amsk∧.≥attrs)⌿6⊃rslt                 ⍝ bin unwanted files and info
    ∇

    ∇ rslt←{amsk}DirX path;handle;next;ok;attrs;⎕IO;FindFirstFileX;FindNextFileX;FindClose;FileTimeToLocalFileTime;FileTimeToSystemTime;GetLastError
     ⍝ Amsk is a 32 element bool attribute mask.
     ⍝ Only files with attributes corresponding to 1-s in the mask will be returned.
     ⍝ Amsk defaults to all attributes.
     ⍝ 0⊃rslt <=> 32 column boolean matrix of attribute bits
     ⍝          [;31] <=> READONLY
     ⍝          [;30] <=> HIDDEN
     ⍝          [;29] <=> SYSTEM
     ⍝          [;28] <=> undocumented
     ⍝          [;27] <=> DIRECTORY
     ⍝          [;26] <=> ARCHIVE
     ⍝          [;25] <=> undocumented
     ⍝          [;24] <=> NORMAL - only set if no other bits are set
     ⍝          [;23] <=> TEMPORARY
     ⍝          [;22] <=> SPARSE FILE
     ⍝          [;21] <=> REPARSE POINT
     ⍝          [;20] <=> COMPRESSED
     ⍝          [;19] <=> OFFLINE
     ⍝          [;18] <=> NOT CONTENT INDEXED
     ⍝          [;17] <=> ENCRYPTED
     ⍝          rest  <=> undocumented
     ⍝ 1⊃rslt <=> 7 column numeric matrix expressing the file creation time in ⎕TS format
     ⍝         if the file system does not support this then all columns are 0
     ⍝ 2⊃rslt <=> 7 column numeric matrix expressing the file last access time in ⎕TS format
     ⍝         if the file system does not support this then all columns are 0
     ⍝ 3⊃rslt <=> 7 column numeric matrix expressing the file last write time in ⎕TS format
     ⍝ 4⊃rslt <=> numeric vector giving the file size accurate up to 53 bits
     ⍝ 5⊃rslt <=> vector of character vectors giving the file names
     ⍝ 6⊃rslt <=> vector of character vectors giving the 8.3 file name for file systems
     ⍝         where it is appropriate and different from the file name
      ⎕IO←0
      :If 900⌶⍬ ⋄ amsk←32⍴1 ⋄ :EndIf
      _FindDefine
      (handle rslt)←_FindFirstFile path
     ('file not found (ntdir error:',(⍕rslt),')')⎕SIGNAL handle↓102   ⍝ file not found
      rslt←,⊂rslt
      :While 1=0⊃ok next←_FindNextFile handle
          rslt,←⊂next
      :EndWhile
      :If 0 18∨.≠ok next
          ('ntdir error:',⍕next)⎕SIGNAL 11   ⍝ DOMAIN
      :EndIf
      ok←FindClose handle
      rslt←↓[0]↑rslt
      (0⊃rslt)←⍉attrs←(32⍴2)⊤0⊃rslt                ⍝ Get attributes into bits
      rslt←(amsk∧.≥attrs)∘⌿¨rslt                ⍝ bin unwanted files and info
      rslt[1 2 3]←↑¨_Filetime_to_TS¨¨rslt[1 2 3]    ⍝ put times into ⎕ts format
      (4⊃rslt)←0(2*32)⊥⍉↑4⊃rslt                    ⍝ combine size elements
      rslt/⍨←5≠⍳8                               ⍝ bin the reserved elements
    ∇

    ∇ r←Exists name
     ⍝ Does file exist?
      r←1
      :Trap 19 22 ⋄ ⎕NUNTIE name ⎕NTIE 0
      :Else ⋄ r←0
      :EndTrap
    ∇

    ∇ r←GetCurrentDirectory;GCD
     ⍝ Get Current Directory using Win32 API
      'GCD'⎕NA'I kernel32.C32|GetCurrentDirectory'A' I4 >0T'
      :If 0≠1⊃r←GCD 256 256
          r←2⊃r
      :Else
          Error'GetCurrentDirectory error:'
      :EndIf
    ∇

    ∇ r←GetText name;tn
     ⍝ Read a text file as single byte text
      tn←name ⎕NTIE 0
      r←⎕NREAD tn(⎕DR' ')(⎕NSIZE tn)
      ⎕NUNTIE tn
    ∇

    ∇ MkDir path;CreateDirectoryA;err;CD
      ⍝ Create a folder using Win32 API
      'CD'⎕NA'I kernel32.C32|CreateDirectory'A' <0T I4' ⍝ Try for best function
      →(0≠CD path 0)⍴0 ⍝ 0 means "default security attributes"
      Error'CreateDirectory error:'
    ∇

    ∇ Move filenames;MV;MX
      'MX'⎕NA'I kernel32.C32|MoveFileEx'A' <0T <0T I4' ⍝ Try for best function
      →0⌿⍨0≠MX filenames,3 ⍝ REPLACE_EXISTING (1) + COPY_ALLOWED (2)
      :Select GetLastError''
      :Case 120                     ⍝ ERROR_CALL_NOT_IMPLEMENTED
          'MV'⎕NA'I Kernel32.C32|MoveFile'A' <0T <0T' ⍝ accept 2nd best - win 95
          →0/⍨0≠MV filenames
      :EndSelect
      Error'MoveFile error:'
    ∇

    ∇ r←text PutText name;tn
     ⍝ Write text to file (must be single byte text)
      :Trap 0
          tn←name ⎕NCREATE 0
      :Else
          tn←name ⎕NTIE 0
          0 ⎕NRESIZE tn
      :EndTrap
      r←text ⎕NAPPEND tn(⎕DR' ')
      ⎕NUNTIE tn
    ∇

    ∇ RmDir path;RM
     ⍝ Remove folder using Win32 API
      'RM'⎕NA'I kernel32.C32|RemoveDirectory'A' <0T'
      →(0≠RM,⊂path)⍴0
      Error'RemoveDirectory error:'
    ∇

    ∇ RmFile file
      file ⎕NERASE file ⎕NTIE tn
    ∇

    ∇ SetCurrentDirectory path;SCD
     ⍝ Set Current Directory using Win32 API
      'SCD'⎕NA'I kernel32.C32|SetCurrentDirectory'A' <0T'
      →(0≠SCD,⊂path)⍴0
      Error'SetCurrentDirectory error:'
    ∇

    ∇ rslt←_Filetime_to_TS filetime;⎕IO
      :If 1≠0⊃rslt←FileTimeToLocalFileTime filetime(⎕IO←0)
      :OrIf 1≠0⊃rslt←FileTimeToSystemTime(1⊃rslt)0
          rslt←0 0                   ⍝ if either call failed then zero the time elements
      :EndIf
      rslt←1 1 0 1 1 1 1 1/1⊃rslt    ⍝ remove day of week
    ∇

    ∇ _FindDefine;WIN32_FIND_DATA
      WIN32_FIND_DATA←'{I4 {I4 I4} {I4 I4} {I4 I4} {U4 U4} {I4 I4} T[260] T[14]}'
      'FindFirstFileX'⎕NA'P kernel32.C32|FindFirstFile'A' <0T >',WIN32_FIND_DATA
      'FindNextFileX'⎕NA'U4 kernel32.C32|FindNextFile'A' I4 >',WIN32_FIND_DATA
      ⎕NA'kernel32.C32|FindClose P'
      ⎕NA'I4 kernel32.C32|FileTimeToLocalFileTime <{I4 I4} >{I4 I4}'
      ⎕NA'I4 kernel32.C32|FileTimeToSystemTime <{I4 I4} >{I2 I2 I2 I2 I2 I2 I2 I2}'
    ∇

    ∇ rslt←_FindFirstFile name;⎕IO
      rslt←FindFirstFileX name(⎕IO←0)
      :If ¯1=0⊃rslt                   ⍝ INVALID_HANDLE_VALUE
          rslt←0,GetLastError''
      :Else
          (1 6⊃rslt)TrimAt←NUL        ⍝ shorten the file name at the null delimiter
          (1 7⊃rslt)TrimAt←NUL        ⍝ and for the alternate name
      :EndIf
    ∇

    ∇ rslt←_FindNextFile handle;⎕IO
      rslt←FindNextFileX handle(⎕IO←0)
      :If 1≠0⊃rslt
          rslt←0,GLE''
      :Else
          (1 6⊃rslt)TrimAt←NUL        ⍝ shorten the filename
          (1 7⊃rslt)TrimAt←NUL        ⍝ shorten the alternate name
      :EndIf
    ∇

    ∇ name←name TrimAt char;⎕IO
     ⍝ Truncates a character vector at character specified.
     ⍝ The character is not included in the result.
      ⎕IO←0 ⋄ name↑⍨←name⍳char
    ∇

    ∇ r←test arg;xyz;tn;fa;fb
      :Trap 283
          MkDir xyz←'/tmp/xyz'
      :Else
          RmDir xyz
          MkDir xyz
      :EndTrap
      tn←(fa←xyz,'/anyname')⎕NCREATE 0
      :Trap 132
          Move fa(fb←xyz,'/newname')
      :EndTrap
      ⎕NUNTIE tn
      Move fa fb
      {}'blah'AppendText fb
      Copy fb fa
      RmFile¨fb fa
      RmDir xyz
    ∇

:EndNamespace ⍝ Files  $Revision: 960 $