﻿:Class WinReg
⍝ Offers shared methods useful to deal with the Windows Registry.
⍝ Note that the Window Registry contains data saved under a kind _
⍝ of "path". Such a path consists of:
⍝ * A so-called "Main Key" like HKEY_CURRENT_USER (or short HKCU)
⍝ * A so called sub key (which is displayed by RegEdt32 as a folder) _
⍝ like "HKEY_CURRENT_USER\Software\Dyalog\Dyalog APL/W 12.0"
⍝ * A so-called value like "maxws"
⍝ These terms might look strange but they are used by this class for _
⍝ the sake of consistency with the Microsoft documentation.
⍝ This class is also able to read and write default values of a _
⍝ sub-key. For that you have to add a "\" to the end of the sub-key.
⍝ Note that the class as such does not come with any limitations _
⍝ regarding the amount of data (REG_SZ, REG_MULTI, REG_EXPAND_SZ, _
⍝ REG_BINARY) you can write. However, Microsoft suggest not to save _
⍝ anything bigger than 2048 bytes (Unicode 4096?!) to avoid performance _
⍝ penalties. One should save larger amounts of data in a file and _
⍝ save the filename rather then the data in the Registry.
⍝ Note that this class supports only a limited number of data types ×
⍝ for writing:
⍝ * REG_SZ (Strings)
⍝ * REG_EXPAND_SZ (STRINGS where variables between % get expanded on read)
⍝ * REG_MULTI_SZ (Vectors of strings)
⍝ * REG_BINARY (Binary data)
⍝ * REG_DWORD (32-bit signed integer)
⍝ Author: Kai Jaeger ⋄ APL Team Ltd ⋄ http://aplteam.com
⍝ Homepage: http://aplwiki.com/WinReg
    ⎕ML←3
    ⎕IO←1

    ∇ R←Version
      :Access Public Shared
      R←'WinReg' '2.2.5' '2011-12-31'
    ∇

    ∇ r←History
      :Access Public Shared
      ⍝ Provides information about the last changes
      r←'See http://aplwiki.com/WinReg/ProjectPage'
    ∇

  ⍝ All data types, including those not supported yet
    :Field Public Shared ReadOnly REG_NONE←0                        ⍝ None
    :Field Public Shared ReadOnly REG_SZ←1                          ⍝ String
    :Field Public Shared ReadOnly REG_EXPAND_SZ←2                   ⍝ String, but somethign like "%WinDir%" is expanded
    :Field Public Shared ReadOnly REG_BINARY←3                      ⍝ Free form binary
    :Field Public Shared ReadOnly REG_DWORD←4                       ⍝ 32-bit number
    :Field Public Shared ReadOnly REG_DWORD_LITTLE_ENDIAN← 4        ⍝ 32-bit number (same as REG_DWORD)
    :Field Public Shared ReadOnly REG_DWORD_BIG_ENDIAN← 5           ⍝ 32-bit number
    :Field Public Shared ReadOnly REG_LINK←6                        ⍝ Symbolic Link (unicode)
    :Field Public Shared ReadOnly REG_MULTI_SZ←7                    ⍝ Multiple Unicode strings
    :Field Public Shared ReadOnly REG_RESOURCE_LIST←8               ⍝ Resource list in the resource map
    :Field Public Shared ReadOnly REG_FULL_RESOURCE_DESCRIPTOR←9    ⍝ Resource list in the hardware description
    :Field Public Shared ReadOnly REG_RESOURCE_REQUIREMENTS_LIST←10
    :Field Public Shared ReadOnly REG_QWORD←11                      ⍝ 64-bit number
    :Field Public Shared ReadOnly REG_QWORD_LITTLE_ENDIAN←11        ⍝ 64-bit number (same as REG_QWORD)
  ⍝ Error codes
    :Field Public Shared ReadOnly ERROR_SUCCESS←0                   ⍝ Return code for success
    :Field Public Shared ReadOnly ERROR_NO_MORE_ITEMS←259           ⍝ No more Values for this SubKey
    :Field Public Shared ReadOnly ERROR_MORE_DATA←234               ⍝ Buffer was too small
    :Field Public Shared ReadOnly ERROR_UNSUPPORTED_TYPE←1630       ⍝ Caution: this is sometimes return when everything is just fine!
    :Field Public Shared ReadOnly ERROR_FILE_NOT_FOUND←2            ⍝ Value not found
    :Field Public Shared ReadOnly ERROR_PATH_NOT_FOUND←3            ⍝ Key not found
    :Field Public Shared ReadOnly ERROR_ACCESS_DENIED←5             ⍝ Insufficient access privileges
    :Field Public Shared ReadOnly ERROR_INVALID_HANDLE←6            ⍝ Handle is invalid
    :Field Public Shared ReadOnly ERROR_NOT_ENOUGH_MEMORY←8         ⍝ Not enough memory
  ⍝ Defines the Access Rights constants
    :Field Public Shared ReadOnly KEY_READ←25
    :Field Public Shared ReadOnly KEY_ALL_ACCESS←983103
    :Field Public Shared ReadOnly KEY_KEY_WRITE←131078
    :Field Public Shared ReadOnly KEY_CREATE_LINK←32
    :Field Public Shared ReadOnly KEY_CREATE_SUB_KEY←4
    :Field Public Shared ReadOnly KEY_ENUMERATE_SUB_KEYS←8
    :Field Public Shared ReadOnly KEY_EXECUTE←131097
    :Field Public Shared ReadOnly KEY_NOTIFE←16
    :Field Public Shared ReadOnly KEY_QUERY_VALUE←1
    :Field Public Shared ReadOnly KEY_SET_VALUE←2
  ⍝ Other stuff
    :Field Public Shared ReadOnly NULL←⎕UCS 0

    ∇ r←{default}GetString y;multiByte;yIsHandle;handle;value;subKey;path;bufSize;∆RegQueryValueEx;rc;type;data;errMsg
      :Access Public Shared
    ⍝ Use this function in order to read a value of type REG_SZ or REG_EXPAND_SZ or REG_MULTI_SZ
    ⍝ y can be one of:
    ⍝ # A simple string which is supposed to be a full path then (sub-key plus value name).
    ⍝ # A vector of length 2 with a handle to the sub-key in [1] and a value name in [2].
      default←{0<⎕NC ⍵:⍎⍵ ⋄ ''}'default'
      r←default
      multiByte←1+80=⎕DR'A' ⍝ 2 in Unicode System
      :If (2=⍴y)∧(0=1↑0⍴1⊃,y)∧(' '=1↑0⍴2⊃,y)
          yIsHandle←1
          (handle value)←y
      :ElseIf 1=≡y
          yIsHandle←0
          path←y
          (subKey value)←{⍵{((-⍵)↓⍺)((-⍵-1)↑⍺)}'\'⍳⍨⌽⍵}path
          subKey←CheckPath subKey
          :If '\'≠¯1↑path
          :AndIf ~DoesKeyExist subKey
              r←default
              :Return
          :EndIf
          handle←OpenKey subKey
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If 0=handle
          r←default
          :Return
      :EndIf
      value←((,'\')≢,value)/value        ⍝ For access to the default value
      '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I =I >0T =I4'
      bufSize←1024
      :Repeat
          (rc type data bufSize)←∆RegQueryValueEx handle value 0 REG_SZ,bufSize bufSize
          :If type=REG_MULTI_SZ ⋄ :Leave ⋄ :EndIf
      :Until rc≠ERROR_MORE_DATA
      :If type=REG_EXPAND_SZ
          data←ExpandEnv data
      :ElseIf type=REG_MULTI_SZ
          '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I =I >T[] =I4'
          :Repeat
              (rc type data bufSize)←∆RegQueryValueEx handle value 0 REG_SZ,bufSize bufSize
              :If type=REG_MULTI_SZ ⋄ :Leave ⋄ :EndIf
          :Until rc≠ERROR_MORE_DATA
          data←Partition(bufSize÷multiByte)↑data
      :EndIf
      errMsg←''
      :If rc=ERROR_FILE_NOT_FOUND
          r←default
      :ElseIf rc=0
          r←data
      :Else
          errMsg←'Erro, rc=',⍕rc
      :EndIf
      Close(~yIsHandle)/handle
      errMsg ⎕SIGNAL 11/⍨~0∊⍴errMsg
    ∇

    ∇ r←{default}GetValue y;yIsHandle;handle;value;subKey;∆RegQueryValueEx;type;rc;errMsg;bufSize;length;data
      :Access Public Shared
    ⍝ y can be either a vector of length one or two:
    ⍝ If length 1:
    ⍝   [1] path (subkey + value name)
    ⍝ If length 2:
    ⍝   [1] handle to the sub key
    ⍝   [2] value name
    ⍝ Returns the data saved as "path" in the Registry, or "Default". The data type is _
    ⍝ determined from the Registry.
      default←{0<⎕NC ⍵:⍎⍵ ⋄ 0}'default'
      r←default
      yIsHandle←0
      :If 2=⍴y
      :AndIf yIsHandle←0=1↑0⍴1⊃y          ⍝ Is the first item possibly a handle?
          (handle value)←y
      :Else
          (subKey value)←SplitPath y
          :If ~DoesKeyExist subKey ⋄ :Return ⋄ :EndIf
          handle←OpenKey subKey
      :EndIf
      '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I >I >I4 =I4'
      :If (,'\')≡,value
          value←''       ⍝ Default value has no value name
      :EndIf
      (rc type)←2↑∆RegQueryValueEx handle value 0 0 0 0
      :If rc=ERROR_FILE_NOT_FOUND
          r←default
          :Return
      :ElseIf ~rc∊ERROR_SUCCESS,ERROR_MORE_DATA,ERROR_UNSUPPORTED_TYPE  ⍝ Yes, I know - but this sometimes happens although everything is fine!!
          errMsg←'Error, rc=',⍕rc
      :EndIf
      errMsg←''
      bufSize←1024
      :Select type
      :Case REG_BINARY
          '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I >I4 >I1[] =I4 '
      :Case REG_DWORD
          '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I >I4 >I4 =I4'
      :CaseList REG_SZ,REG_EXPAND_SZ,REG_MULTI_SZ
⍝          '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I >I4 >0T =I4'
          '∆RegQueryValueEx'⎕NA'I ADVAPI32.dll.C32|RegQueryValueEx',AnsiOrWide,' U <0T I =I >T[] =I4'
      :Else
          ('Unsupported data type: ',GetTypeAsString type)⎕SIGNAL 11
      :EndSelect
      :Repeat
          (rc type data length)←∆RegQueryValueEx handle value 0 1024 bufSize 1024
          bufSize+←1024
      :Until ERROR_MORE_DATA≠rc
      errMsg←''
      :If rc=ERROR_FILE_NOT_FOUND
          r←default
      :ElseIf ~rc∊ERROR_SUCCESS,ERROR_UNSUPPORTED_TYPE  ⍝ Yes, I know - but this sometimes happens although everything is fine!!
          errMsg←'Error, rc=',⍕rc
      :Else
          r←HandleDataType type length data
      :EndIf
      Close(~yIsHandle)/handle
      errMsg ⎕SIGNAL 11/⍨~0∊⍴errMsg
    ∇

    ∇ {r}←{type}PutString y;yIsHandle;path;data;value;subKey;handle;multiByte;rc
      :Access Public Shared
    ⍝ y can be either a vector of length two or three:
    ⍝ If length 2:
    ⍝   [1] path (subkey + value name)
    ⍝   [2] data to be saved
    ⍝ If length 3:
    ⍝   [1] handle to the sub key
    ⍝   [2] value name
    ⍝   [3] data to be saved
    ⍝ Stores "data" under ¯1↓y. If "path" ends with a "\" char, "data" _
    ⍝ is saved as the default value of "path"; data type is always "REG_SZ" then.
    ⍝ Note that type defaults to "REG_SZ" except when "data" is nested, then _
    ⍝ the default is "REG_MULTI_SZ.
    ⍝ You can set "type" to one of: REG_SZ, REG_EXPAND_SZ, REG_MULTI_SZ
      :Select ↑⍴,y
      :Case 2
          yIsHandle←0
          (path data)←y
          (subKey value)←{⍵{((-⍵)↓⍺)((-⍵-1)↑⍺)}'\'⍳⍨⌽⍵}path
          subKey←CheckPath subKey
          handle←OpenAndCreateKey subKey
      :Case 3
          yIsHandle←1
          (handle value data)←y
          'Invalid right argument'⎕SIGNAL 11/⍨0≠1↑0⍴handle
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndSelect
      data←,data
      multiByte←1+80=⎕DR'A' ⍝ 2 in Unicode System
      'Invalid "value"'⎕SIGNAL 11/⍨' '≠1↑0⍴value
      type←data{2=⎕NC ⍵:⍎⍵ ⋄ (1+0 1∊⍨≡⍺)⊃REG_MULTI_SZ REG_SZ}'type'
      :If type=REG_MULTI_SZ
          data←{0 1∊⍨≡⍵:⍵ ⋄ 0∊⍴⍵:'' ⋄ (↑,/⍵,¨NULL),NULL}data
      :Else
          data←data,{NULL=⍵:'' ⋄ NULL}¯1↑data
      :EndIf
      'Invalid data type - must be one of REG_SZ,REG_MULTI_SZ,REG_EXPAND_SZ'⎕SIGNAL 11/⍨~type∊REG_SZ,REG_MULTI_SZ,REG_EXPAND_SZ
      '∆RegSetValueEx'⎕NA'I ADVAPI32.dll.C32|RegSetValueEx',AnsiOrWide,' I <0T I I <T[] I'
      :If yIsHandle
      :AndIf (,'\')≡,value
          rc←∆RegSetValueEx handle'' 0 type data(multiByte×↑⍴data)
      :Else
          rc←∆RegSetValueEx handle value 0 type data(multiByte×↑⍴data)
      :EndIf
      ('Error, rc=',⍕rc)⎕SIGNAL 11/⍨ERROR_SUCCESS≠rc
      Close(~yIsHandle)/handle
      r←⍬
    ∇

    ∇ {r}←PutValue y;yIsHandle;data;path;subKey;value;handle;multiByte;∆RegSetValueEx
      :Access Public Shared
    ⍝ Stores "data" under "path". Note that you cannot save a default value _
    ⍝ with "PutValue" (=path ends with a backslash) because default values _
    ⍝ MUST be of type REG_SZ. Therefore use "PutString" in order to set a _
    ⍝ default value. If path ends with a backslash, an exception is thrown.
    ⍝ Note that you can only save REG_DWORDs with PutVales, that is 32-bit _
    ⍝ integers.
      :Select ↑⍴,y
      :Case 2
          yIsHandle←0
          (path data)←y
          'Default values must be of type REG_SZ (string)'⎕SIGNAL 11/⍨'\'=¯1↑path
          (subKey value)←SplitPath path
          subKey←CheckPath subKey
          handle←OpenAndCreateKey subKey
      :Case 3
          yIsHandle←1
          (handle value data)←y
          'Invalid right argument'⎕SIGNAL 11/⍨0≠1↑0⍴handle
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndSelect
      data←,data
     ⍝ multiByte←1+80=⎕DR'A' ⍝ 2 in Unicode System
      '∆RegSetValueEx'⎕NA'I ADVAPI32.dll.C32|RegSetValueEx',AnsiOrWide,' I <0T I I <I4 I'
      :If (1≠≡data)∨1≠⍴data←,data
          Close(~yIsHandle)/handle
          'Data has invalid depth/shape'⎕SIGNAL 11
      :EndIf
      :If data>2147483647
          Close(~yIsHandle)/handle
          'Data too large; max is 2,147,483,647'⎕SIGNAL 11
      :EndIf
      :If data<¯2147483647
          Close(~yIsHandle)/handle
          'Data too small; min is -2,147,483,647'⎕SIGNAL 11
      :EndIf
      _←∆RegSetValueEx handle value 0 REG_DWORD data 4
      Close(~yIsHandle)/handle
      r←⍬
    ∇

    ∇ {r}←PutBinary y;yIsHandle;path;data;value;subKey;handle;multiByte;∆RegSetValueEx
      :Access Public Shared
    ⍝ Stores binary "data".
    ⍝ y may be one of:
    ⍝ # A vector of length 2 with the full path (sub-key + value name) in _
    ⍝   the first item and the data in the second item.
    ⍝ # A vector of length 3 with a handle in [1], the value name in [2] _
    ⍝   and the data in [3].
    ⍝ Note that you cannot save a default _
    ⍝ value with "PutBinary" (=path ends with a backslash) because default _
    ⍝ values MUST be of type REG_SZ. Therefore use "PutString" in order to _
    ⍝ set a default value. If path ends with a backslash, an exception is thrown.
    ⍝ Note that for binary you must specify Integer (¯128 to 128).
      :Select ↑⍴,y
      :Case 2
          yIsHandle←0
          (path data)←y
          'Default values must be of type REG_SZ (string)'⎕SIGNAL 11/⍨'\'=¯1↑path
          (subKey value)←SplitPath path
          subKey←CheckPath subKey
          handle←OpenAndCreateKey subKey
      :Case 3
          yIsHandle←1
          (handle value data)←y
          'Default values must be of type REG_SZ (string)'⎕SIGNAL 11/⍨(,'\')≡,value
          'Invalid right argument'⎕SIGNAL 11/⍨0≠1↑0⍴handle
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndSelect
⍝      multiByte←1+80=⎕DR'A' ⍝ 2 in Unicode System
      data←,data
      '∆RegSetValueEx'⎕NA'I ADVAPI32.dll.C32|RegSetValueEx',AnsiOrWide,' I <0T I I <I1[] I'
      _←∆RegSetValueEx handle value 0 REG_BINARY data(↑⍴data)
      Close(~yIsHandle)/handle
      r←⍬
    ∇

    ∇ r←KeyInfo y;yIsHandle;handle;∆RegQueryInfoKey;buffer;rc;noofValues;noofSubKeys;maxNameLength;maxValueLength
    ⍝ Returns a vector with information about the Key (not Value!) in question
    ⍝ # No. of values
    ⍝ # No. of subkeys
    ⍝ # Largest length of all value names (without \n)
    ⍝ # Largest length of value data (without \n)
    ⍝ y can be one of:
    ⍝ # A handle
    ⍝ # A path (sub key)
      :Access Public Shared
      :If (0=1↑0⍴y)∧1=⍴,y       ⍝ Is it a handle?
          yIsHandle←1
          handle←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          handle←OpenKey y
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      '∆RegQueryInfoKey'⎕NA'I ADVAPI32|RegQueryInfoKey',AnsiOrWide,' I >T[] =I I >I >I >I >I >I >I >I >{U U}'
      buffer←∆RegQueryInfoKey handle 0 0 0 1 1 1 1 0 0 0 0
      rc←1⊃buffer
      :If 0=rc
          noofValues←7⊃buffer
          noofSubKeys←4⊃buffer
          maxNameLength←8⊃buffer
          maxValueLength←9⊃buffer
          r←noofValues noofSubKeys maxNameLength maxValueLength
      :Else
          Close(~yIsHandle)/handle
          ('Error, rc=',⍕rc)⎕SIGNAL 11
      :EndIf
      Close(~yIsHandle)/handle
    ∇

    ∇ r←GetAllValues y;names;handle;noof;i;yIsHandle
    ⍝ This method gets all values for a given subkey
    ⍝ r is a matrix with:
    ⍝ [;1] Value name
    ⍝ [;2] The data
    ⍝ y can be one of:
    ⍝ # A string representing a path (sub key)
    ⍝ # A handle
      :Access Public Shared
      :If (0=1↑0⍴y)∧1=⍴,y       ⍝ Is it a handle?
          yIsHandle←1
          handle←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          handle←OpenKey y
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If 0=handle
          r←0 2⍴⍬
          :Return
      :EndIf
      noof←⍴names←GetAllValueNames handle
      r←(noof,2)⍴⍬
      :If ~0∊⍴r
          r[;1]←names
          r[;2]←{GetValue handle ⍵}¨names
      :EndIf
      Close(~yIsHandle)/handle
    ∇

    ∇ r←{verbose}GetAllValueNames y;yIsHandle;handle;noofValues;noofSubkeys;For;∆RegEnumValue;i;rc;data;length;type;keyLength;dataLength
    ⍝ This method gets all value names for a given subkey.
    ⍝ r is a vector with value names or, if the left argument is the string _
    ⍝ "verbose" (default is ''), a matrix with 2 columns:
    ⍝ [;1] names
    ⍝ [;2] data types
    ⍝ y can be one of:
    ⍝ # A string which is treated as a path (sub key)
    ⍝ # An integer which is treated as a handle
    ⍝ Note that for a default value a "\" is returned as value name.
      :Access Public Shared
      verbose←{2=⎕NC ⍵:'verbose'≡⍎⍵ ⋄ 0}'verbose'
      :If (0=1↑0⍴y)∧1=⍴,y       ⍝ Is it a handle?
          yIsHandle←1
          handle←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          handle←OpenKey y
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      '∆RegEnumValue'⎕NA'I ADVAPI32|RegEnumValue',AnsiOrWide,' I I >T[] =I I >I >T[] =I'
      (noofValues noofSubkeys keyLength dataLength)←4↑KeyInfo handle  ⍝ No. of values, no. of SubKeys, max name length, data length
      r←(noofValues,2)⍴' '
      :For i :In (⍳noofValues)-1
          (rc data length type)←4↑∆RegEnumValue handle,i,(keyLength+1),(keyLength+1),0 1,(dataLength+1),(dataLength+1)
          :If rc∊ERROR_SUCCESS,ERROR_MORE_DATA
              :If 0=length      ⍝ Then it is the default value
                  r[i+1;]←'\'
              :Else
                  r[i+1;]←(⊃↑/length data)type
              :EndIf
          :Else
              Close(~yIsHandle)/handle
              ('Error, rc=',⍕rc)⎕SIGNAL 11
          :EndIf
      :EndFor
      Close(~yIsHandle)/handle
      :If ~verbose
          r←r[;1]
      :EndIf
    ∇

    ∇ r←GetAllSubKeyNames y;yIsHandle;handle;∆RegEnumKey;flag;rc;i;bufSize;name;length
    ⍝ This method returns a vector of strings with the namesof all sub keys for a given key.
      :Access Public Shared
      :If (0=1↑0⍴y)∧1=⍴,y       ⍝ Is it a handle?
          yIsHandle←1
          handle←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          handle←OpenKey y
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      '∆RegEnumKey'⎕NA'I ADVAPI32|RegEnumKeyEx',AnsiOrWide,'  I I >T[] =I I I I I'
      flag←i←0
      bufSize←1024×1+80=⎕DR''
      r←''
      :Repeat
          (rc name length)←∆RegEnumKey handle,i,bufSize,bufSize,0 0 0 0
          :If rc=ERROR_SUCCESS
              r,←⊂length↑name
              i+←1
          :ElseIf rc=ERROR_NO_MORE_ITEMS
              flag←1
              Close(~yIsHandle)/handle
          :ElseIf rc=ERROR_INVALID_HANDLE
              r←''
              flag←1
          :Else
              Close(~yIsHandle)/handle
              ('Error, rc=',⍕rc)⎕SIGNAL 11
          :EndIf
      :Until flag
      Close(~yIsHandle)/handle
    ∇

    ∇ r←{depth}GetTree key;handle;depth;allValues;AllSubKeys;thisSubKey;buffer;allSubKeys
    ⍝ Takes the name of a key (but no handle!) and returns a (possibly empty) matrix with:
    ⍝ [;1] depth
    ⍝ [;2] fully qualified name
    ⍝ Note that sub-keys end with a backslash.
    ⍝ See "GetTreeWithValues" if you need the data of the values, too.
      :Access Public Shared
      depth←{0=⎕NC ⍵:0 ⋄ ⍎⍵}'depth'
      :If (0=1↑0⍴key)∧1=⍴,key                   ⍝ Is it a valid handle?
          11/⍨⎕SIGNAL'Invalid right argument: must be a key name'
      :ElseIf (' '=1↑0⍴key)∧1=≡key              ⍝ Is it a path?
          handle←OpenKey key
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If handle=0
          11 ⎕SIGNAL⍨'Key "',key,'" does not exist'
      :Else
          r←1 2⍴depth(key,('\'≠¯1↑key)/'\')
          :If ~0∊⍴allValues←GetAllValueNames handle
              r⍪←(1+depth),[1.5](⊂key,('\'≠¯1↑key)/'\'),¨allValues
          :EndIf
          :If ~0∊⍴allSubKeys←GetAllSubKeyNames key
              :For thisSubKey :In allSubKeys
                  :If ~0∊⍴buffer←(depth+1)GetTree key,(('\'≠¯1↑key)/'\'),thisSubKey
                      r⍪←buffer
                  :EndIf
              :EndFor
          :EndIf
          Close handle
      :EndIf
    ∇

    ∇ r←{depth}GetTreeWithValues key;handle;depth;allValues;AllSubKeys;thisSubKey;buffer;allSubKeys
    ⍝ Takes the name of a key (but no handle!) and returns a (possibly empty) matrix with:
    ⍝ [;1] depth
    ⍝ [;2] fully qualified name
    ⍝ [;3] value data (empty for sub-keys)
    ⍝ Note that sub-keys end with a backslash.
    ⍝ See "GetTree" if you don't need the data of the values.
      :Access Public Shared
      depth←{0=⎕NC ⍵:0 ⋄ ⍎⍵}'depth'
      :If (0=1↑0⍴key)∧1=⍴,key                   ⍝ Is it a valid handle?
          11/⍨⎕SIGNAL'Invalid right argument: must be a key name'
      :ElseIf (' '=1↑0⍴key)∧1=≡key              ⍝ Is it a path?
          handle←OpenKey key
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If handle=0
          11 ⎕SIGNAL⍨'Key "',key,'" does not exist'
      :Else
          r←1 3⍴depth(key,('\'≠¯1↑key)/'\')''
          :If ~0∊⍴allValues←GetAllValues handle
              allValues[;1]←(⊂key,('\'≠¯1↑key)/'\'),¨allValues[;1]
              r⍪←(1+depth),allValues
          :EndIf
          :If ~0∊⍴allSubKeys←GetAllSubKeyNames key
              :For thisSubKey :In allSubKeys
                  :If ~0∊⍴buffer←(depth+1)GetTreeWithValues key,(('\'≠¯1↑key)/'\'),thisSubKey
                      r⍪←buffer
                  :EndIf
              :EndFor
          :EndIf
          Close handle
      :EndIf
    ∇

    ∇ {r}←CopyTree(source destination);sourceHandle;destinationHandle;wv;sourceIsHandle;destinationIsHandle;∆RegCopyTree
      :Access Public Shared
     ⍝ Use this to copy a Registry Key from "source" to "destination".
     ⍝ Note that this method needs at least Vista.
     ⍝ Both, "source" as well as "destination" can be one of:
     ⍝ # A path (sub key)
     ⍝ # A handle
      :If (0=1↑0⍴source)∧1=⍴,source       ⍝ Is it a handle?
          sourceIsHandle←1
          sourceHandle←source
      :ElseIf (' '=1↑0⍴source)∧1=≡source  ⍝ Is it a path
          sourceHandle←OpenKey source
          sourceIsHandle←0
      :Else
          'Invalid right argument: "source"'⎕SIGNAL 11
      :EndIf
      :If (0=1↑0⍴destination)∧1=⍴,destination       ⍝ Is it a handle?
          destinationIsHandle←1
          destinationHandle←destination
      :ElseIf (' '=1↑0⍴destination)∧1=≡destination  ⍝ Is it a path
          destinationHandle←OpenAndCreateKey destination
          destinationIsHandle←0
      :Else
          'Invalid right argument: "source"'⎕SIGNAL 11
      :EndIf
      :If ~destinationIsHandle
          ('Destination must not be a Registry Value: ',source)⎕SIGNAL 11/⍨0≠#.WinReg.DoesValueExist destination
      :EndIf
      wv←GetVersion                 ⍝ Get the Windows version
      '"CopyTree" is not supported in this version of Windows'⎕SIGNAL 11/⍨6>1⊃wv
      'Recursive copy'⎕SIGNAL 11/⍨source≡(⍴source)↑destination
      '∆RegCopyTree'⎕NA'I ADVAPI32.dll.C32|RegCopyTree',AnsiOrWide,' U <0T[] U I4'
      r←∆RegCopyTree sourceHandle''destinationHandle 0
      Close(~sourceIsHandle)/sourceHandle
      Close(~destinationIsHandle)/destinationHandle
    ∇

    ∇ {r}←DeleteSubKey y;handle;HKEY;subKey;∆RegDeleteKey;wv;yIsHandle;path
      :Access Public Shared
     ⍝ Deletes a subkey "path", even if this subkeys holds values.
     ⍝ The subkey to be deleted must not have subkeys. (You can achieve _
     ⍝ this with the "DeleteSubKeyTree" function, see there)
     ⍝ y can be one of:
     ⍝ # A string which is treated as a path (sub key)
     ⍝ # An integer which is treated as a handle
     ⍝ Note that for a default value a "\" is returned as value name.
      :Access Public Shared
      :If (0=1↑0⍴y)∧1=⍴,y       ⍝ Is it a handle?
          handle←y
          subKey←''
          yIsHandle←1
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          y←CheckPath y
          (path subKey)←SplitPath y
          handle←OpenKey path
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If 0=handle ⋄ r←0 ⋄ :Return ⋄ :EndIf  ⍝ handle is 0? Nothing to delete then.
      wv←GetVersion                 ⍝ Get the Windows version
      '"DeleteSubKey" is not supported in this version of Windows'⎕SIGNAL 11/⍨6>1⊃wv
      '∆RegDeleteKey'⎕NA'I ADVAPI32.dll.C32|RegDeleteKey',AnsiOrWide,' U <0T[]'
      r←∆RegDeleteKey handle subKey
      Close(~yIsHandle)/handle
    ∇

    ∇ {r}←DeleteSubKeyTree y;handle;∆RegDeleteTree;yIsHandle;subKey;path
      :Access Public Shared
     ⍝ Deletes a subkey "path", even if this subkeys holds values.
     ⍝ Any subkeys in "path" will be deleted as well.
     ⍝ Note that this methods needs at least Vista.
     ⍝ y can be one of:
     ⍝ # A path (sub key)
     ⍝ # A handle to a sub key
      'Right argument must not be empty'⎕SIGNAL 11/⍨0∊⍴y
      :If 1≠≡y
      :AndIf (0=1↑0⍴1⊃,y)∧1=⍴,1⊃,y       ⍝ Is it a handle?
          yIsHandle←1
          handle←y
          subKey←''
      :ElseIf (' '=1↑0⍴y)∧1=≡y           ⍝ Is it a path?
          y←CheckPath y
          (path subKey)←SplitPath{⍵↓⍨-'\'=¯1↑⍵}y
          handle←OpenKey path
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      '∆RegDeleteTree'⎕NA'I ADVAPI32.dll.C32|RegDeleteTree',AnsiOrWide,' U <0T[]'
      r←∆RegDeleteTree handle subKey
      Close(~yIsHandle)/handle
    ∇

    ∇ {r}←DeleteValue y;path;RegDeleteValueA;handle;∆RegDeleteValue;value;subKey;yIsHandle
      :Access Public Shared
     ⍝ Delete a value from the Windows Registry.
     ⍝ y can be one of:
     ⍝ [1] A full path (sub key + value name)
     ⍝ [2] A vector of length 2 with a handle in [1] and a value name in [2]
     ⍝ This method normally returns either ERROR_SUCCESS or ERROR_FILE_NOT_FOUND ×
     ⍝ in case the value did not exist from the start.
      :If 1≠≡y
      :AndIf (0=1↑0⍴1⊃,y)∧1=⍴,1⊃,y       ⍝ Is it a handle?
          yIsHandle←1
          (handle value)←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          (subKey value)←SplitPath y
          subKey←CheckPath subKey
          handle←OpenKey subKey
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      '∆RegDeleteValue'⎕NA'I ADVAPI32.dll.C32|RegDeleteValue',AnsiOrWide,' U <0T[]'
      r←∆RegDeleteValue handle value
      Close(~yIsHandle)/handle
    ∇

    ∇ bool←DoesKeyExist path;handle
      :Access Public Shared
    ⍝ Checks if a given Registry key exists
    ⍝ Note that you cannot pass a handle as right argument because _
    ⍝ that makes no sense: if there is a handle the sub key MUST exist.
      'Right argument must not be empty'⎕SIGNAL 11/⍨0∊⍴path
      handle←OpenKey path
      bool←handle≠0
      Close handle
    ∇

    ∇ r←GetTypeAsStringFrom value;l;values;mask
    ⍝ Use this to convert a value like 3 to REG_BINARY or 'REG_BINARY' _
    ⍝ into 3. Specify an empty argument to get a matrix with all possible _
    ⍝ names and their values.
      :Access Public Shared
      'Invalid right argument'⎕SIGNAL 11/⍨~0 1∊⍨≡value
      l←GetAllTypes ⍬            ⍝ Get list with all REG fields
      r←l GetAsString value
    ∇

    ∇ r←GetErrorAsStringFrom value;l;values;mask
    ⍝ Use this to convert a value like 8 to ERROR_NOT_ENOUGH_MEMORY
    ⍝ into 3. Specify an empty argument to get a matrix with all possible _
    ⍝ names and their values.
      :Access Public Shared
      'Invalid right argument'⎕SIGNAL 11/⍨~0 1∊⍨≡value
      l←GetAll'ERROR'               ⍝ Get list with all errors
      r←l GetAsString value
    ∇

    ∇ bool←DoesValueExist y;names;yIsHandle;value;handle;subKey
      :Access Public Shared
    ⍝ Checks if a value exists in the Registry.
    ⍝ y can be one of:
    ⍝ # Path (sub key)
    ⍝ # A handle to a sub key
      :If 1≠≡y
      :AndIf (0=1↑0⍴1⊃,y)∧1=⍴,1⊃,y       ⍝ Is it a handle?
          yIsHandle←1
          (handle value)←y
      :ElseIf (' '=1↑0⍴y)∧1=≡y  ⍝ Is it a path
          (subKey value)←SplitPath y
          subKey←CheckPath subKey
          handle←OpenKey subKey
          yIsHandle←0
      :Else
          'Invalid right argument'⎕SIGNAL 11
      :EndIf
      :If 0=handle ⍝ Then the sub key does not exist, let alone the value
          bool←0
      :Else
          names←GetAllValueNames handle
          bool←(⊂Lowercase value)∊Lowercase names
          Close(~yIsHandle)/handle
      :EndIf
    ∇

    ∇ Close handle;RegCloseKey;_
      :Access Public Shared
      :If ~0∊⍴handle
          ⎕NA'U ADVAPI32.dll.C32|RegCloseKey U'
          _←RegCloseKey¨handle
      :EndIf
    ∇

    ∇ handle←{accessRights}OpenKey path;HKEY;subKey;∆RegCreateKeyEx;rc
    ⍝ Opens a key. This will fail if the key does not already exist.
    ⍝ See also: OpenKeyAndCreate
    ⍝ The optional left argument defaults to KEY_ALL_ACCESS which includes "Create"). _
    ⍝ Instead you can specify KEY_READ in case of lacking the rights to create anything.
      :Access Public Shared
      accessRights←{2=⎕NC ⍵:⍎⍵ ⋄ KEY_ALL_ACCESS}'accessRights'
      path←CheckPath path
      :If 'HKEY_'≡5↑path
          (HKEY subKey)←{a←⍵⍳'\' ⋄ ((a-1)↑⍵)(a↓⍵)}path
          HKEY←Get_HKEY_From HKEY
      :Else
          HKEY←Get_HKEY_From'HKEY_CURRENT_USER'  ⍝ Default
      :EndIf
      '∆RegCreateKeyEx'⎕NA'I ADVAPI32.dll.C32|RegOpenKeyEx',AnsiOrWide,' U <0T I I >I'
      (rc handle)←∆RegCreateKeyEx HKEY subKey 0 accessRights 1
      ('Error, rc=',⍕rc)⎕SIGNAL 11/⍨~rc∊ERROR_SUCCESS,ERROR_FILE_NOT_FOUND
    ∇

    ∇ handle←OpenAndCreateKey path;HKEY;subKey;∆RegCreateKeyEx;rc;newFlag
    ⍝ Opens a key. If the key does not already exist it is going to be created.
    ⍝ See also: OpenKey
    ⍝ Note that this method needs KEY_ALL_ACCESS otherwise it cannot work properly.
      :Access Public Shared
      path←CheckPath path
      :If 'HKEY_'≡5↑path
          (HKEY subKey)←{a←⍵⍳'\' ⋄ ((a-1)↑⍵)(a↓⍵)}path
          HKEY←Get_HKEY_From HKEY
      :Else
          HKEY←Get_HKEY_From'HKEY_CURRENT_USER'  ⍝ Default
      :EndIf
      '∆RegCreateKeyEx'⎕NA'I ADVAPI32.dll.C32|RegCreateKeyEx',AnsiOrWide,' U <0T I <0T I I I >U >U'
      (rc handle newFlag)←∆RegCreateKeyEx HKEY subKey 0 '' 0 KEY_ALL_ACCESS 0 1 1
      ('Error, rc=',⍕rc)⎕SIGNAL 11/⍨ERROR_SUCCESS≠rc
    ∇


                            ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝ Private methods ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝

    ∇ HKEY←Get_HKEY_From Type
      Type←{0∊⍴⍵:'HKEY_CURRENT_USER' ⋄ ⍵}Type
      :If ' '=1↑0⍴Type
          :Select Type
          :Case 'HKEY_CLASSES_ROOT'
              HKEY←2147483648             ⍝ HEX 0x80000000
          :Case 'HKEY_CURRENT_USER'
              HKEY←2147483649             ⍝ HEX 0x80000001
          :Case 'HKEY_LOCAL_MACHINE'
              HKEY←2147483650             ⍝ HEX 0x80000002
          :Case 'HKEY_USERS'
              HKEY←2147483651             ⍝ HEX 0x80000003
          :Case 'HKEY_PERFORMANCE_DATA'
              HKEY←2147483652             ⍝ HEX 0x80000004
          :Case 'HKEY_CURRENT_CONFIG'
              HKEY←2147483653             ⍝ HEX 0x80000050
          :Case 'HKEY_DYN_DATA'
              HKEY←2147483654             ⍝ HEX 0x80000060
          :Else
              'Invalid Keyword'⎕SIGNAL 11
          :EndSelect
      :Else
          HKEY←Type
      :EndIf
    ∇

    ∇ path←CheckPath path;buffer;path2;HKEY
   ⍝ Check the path, replace shortcuts by proper names and establish default if needed
      :If 'HK'≡2↑path
          (HKEY path2)←{⍵{((¯1+⍵)↑⍺)(⍵↓⍺)}⍵⍳'\'}path
          :If 'HKEY_'{⍺≢⍵↑⍨⍴⍺}HKEY
              :Select HKEY
              :Case 'HKCU'
                  path←'HKEY_CURRENT_USER\',path2
              :Case 'HKCR'
                  path←'HKEY_CLASSES_ROOT\',path2
              :Case 'HKLM'
                  path←'HKEY_LOCAL_MACHINE\',path2
              :Case 'HKU'
                  path←'HKEY_USERS\',path2
              :Else
                  11 ⎕SIGNAL⍨'Invalid Registry key: "',HKEY,'"'
              :EndSelect
          :EndIf
      :Else
          path←'HKEY_CURRENT_USER\',path
      :EndIf
    ∇

      Make←{
          0=⎕NC ⍵:⍺
          ⍎⍵}

      GetAll←{
      ⍝ Examples:
      ⍝ GetAll 'REG'
      ⍝ GetAll 'ERROR'
          l←⎕NL 2                           ⍝ All variables
          l←(l[;⍳⍴,⍵]∧.=⍵)⌿l                ⍝ Only those which start with ⍵
          (↓l)~¨' '                         ⍝ Transform into vectors of vectors
      }

    Partition←{⍵⊂⍨⍵≠⎕ucs 0}

    ∇ r←GetVersion;rc;OSVERSIONINFO
   ⍝ Gets the OS version.
   ⍝ r[0] = Major version
   ⍝ r[1] = Minor version
   ⍝ r[2] = BuildNumber
      '∆GetVersion'⎕NA'I KERNEL32|GetVersionExA ={I I I I I T[128]}' ⍝ Don't convert this to Unicode - no point!
      OSVERSIONINFO←148 0 0 0 0(128⍴NULL)
      (rc r)←∆GetVersion⊂OSVERSIONINFO
      r←3↑1↓r
    ∇

    ∇ R←ExpandEnv Y;ExpandEnvironmentStrings;multiByte
    ⍝ If Y does not contain any "%", Y is passed untouched.
    ⍝ In case Y is empty R is empty as well.
    ⍝ Example:
    ⍝ <pre>'C:\Windows\MyDir' ←→ #.WinSys.ExpandEnv '%WinDir%\MyDir'</pre>
      :If '%'∊R←Y
          'ExpandEnvironmentStrings'⎕NA'I4 KERNEL32.C32|ExpandEnvironmentStrings',('*A'⊃⍨1+12>{⍎⍵↑⍨1+⍵⍳'.'}2⊃'.'⎕WG'APLVersion'),' <0T >0T I4'
          multiByte←1+80=⎕DR' '       ⍝ Unicode version? (used to double the buffer size)
          R←2⊃ExpandEnvironmentStrings(Y(multiByte×1024)(multiByte×1024))
      :EndIf
    ∇

    ∇ r←HandleDataType(type length data);multiByte
      multiByte←80=⎕DR' '
      :Select type
      :Case REG_SZ
          r←(¯1+length÷1+multiByte)↑data
      :Case REG_DWORD
          r←data
      :Case REG_BINARY
          r←length↑data
      :Case REG_EXPAND_SZ
          r←ExpandEnv(¯1+length÷1+multiByte)↑data
      :Case REG_MULTI_SZ
          r←Partition(¯1+length÷1+multiByte)↑data
      :EndSelect
    ∇

    ∇ r←AnsiOrWide;⎕IO
      ⎕IO←0
      r←'*A'⊃⍨12>{⍎⍵↑⍨⍵⍳'.'}1⊃'.'⎕WG'APLVersion'
    ∇

    ∇ r←mat GetAsString value;values
    ⍝ Sub fns of GetErrorAsStringFrom and GetTypeAsStringFrom
      :If 0∊⍴value
          r←⍉mat,[0.5]⍎¨mat             ⍝ Build matrix with cols "Name" and "Value"
      :Else
          values←⍎¨l                ⍝ Get the values
          r←{1=⍴,⍵:↑⍵ ⋄ ⍵}(mat,⊂'?')[values⍳value]
      :EndIf
    ∇


    :Section APLTreeUtils                                

⍝ Homepage: http://aplwiki.com/APLTreeUtils
⍝
⍝ *** Version 2.3.0 ⋄ 2012-03-018 ***
⍝ 2.3.0  "ReadUtf8File" now accepts a left argument "flat".
⍝ 2.2.3  "Nest" transformed a nested scalar into a vector; it shoudn't.
⍝ 2.2.2  The "Trans" operator can produce a WS FULL that dispappears when →⎕LC is executed.
⍝        This version gets around those errors by executing the statement again but just once.
⍝ 2.2.1: Bug in dtb with matrices fixed.
⍝ 2.2.0: New methods "dlb" and "dtb".
⍝        Fix: result of "Where" should depend on caller's ⎕IO but didn't.
⍝ 2.1.0: New method "Where" ⋄ New parameter for WriteUtf8File ↓ ... but only if the file is not empty!
⍝ 2.0.3: WriteUtf8File should add a leading pair of (⎕UCS 13 10) whith "append" when the data is nested.
⍝ 2.0.2: The optional left argument "append" was ignored by WriteUtf8File.
⍝ 2.0.1: Method "GoToWebPage" now also accepts URLs like file://foo.html". Works not everywhere I am adraid.
⍝ 1.9.1: Bug in ReadUtf8File and WriteUtf8File regarding double quotes.
⍝ 1.9.0: Copyright notice removed altogether. New methods Enlist and IsUnicode
⍝ 1.8.0: Copyright changed: no conditions/obligations
⍝ 1.7.0: Renamed to "APLTreeUtils"
⍝ 1.6.0: "Copyright" method has changed: no restrictions anymore.
⍝
⍝ Needs Dyalog Version 12, Unicode for "WriteUtf8File" & "ReadUtf8File"
⍝ From version 1.32 on, all fns in this script are tested to be _
⍝ independent from ⎕IO and ⎕ML except when declared to be different (Where..,)

    ∇ array←Uppercase array;ToUpper;⎕ML;⎕IO
      ⎕IO←⎕ML←1
      :If ~0∊⍴array
          'ToUpper'⎕NA'I4 USER32.C32|CharUpper',('*A'⊃⍨1+12>{⍎⍵↑⍨¯1+⍵⍳'.'}2⊃'.'⎕WG'APLVersion'),' =0T'
          :If ~0 1∊⍨≡array
              array←(ToUpper Trans)¨array
          :Else
              array←(ToUpper Trans)array
          :EndIf
      :EndIf
    ∇

    ∇ array←Lowercase array;ToLower;⎕ML;⎕IO
      ⎕IO←⎕ML←1
      :If ~0∊⍴array
          'ToLower'⎕NA'I4 USER32.C32|CharLower',('*A'⊃⍨1+12>{⍎⍵↑⍨¯1+⍵⍳'.'}2⊃'.'⎕WG'APLVersion'),' =0T'
          :If ~0 1∊⍨≡array
              array←(ToLower Trans)¨array
          :Else
              array←(ToLower Trans)array
          :EndIf
      :EndIf
    ∇

      Trans←{⎕IO←1
      ⍝ This operator can produce a WS FULL that often disappears when →⎕LC is executed.
      ⍝ Therefore it calles itself via a WS FULL guard but just once.
          ⍺←0               ⍝ The default.
          ⍺:(⍴⍵)⍴2⊃⍺⍺⊂,⍵    ⍝ If ⍺ is true then by now a workspace compaction was performed.
          1::1 ∇ ⍵          ⍝ In case of a WS FULL call itself with a 1 as left argument.
          (⍴⍵)⍴2⊃⍺⍺⊂,⍵      ⍝  Execute it.
      }

    IsChar ←{0 2∊⍨10|⎕dr ⍵}  ⍝ Version 12 compatible: is 82=⎕dr in V11

    ∇ r←IsUnicode
    ⍝ Return a 1 if running under Dyalog Unicode
      r←80=⎕DR' '
    ∇

      SplitPath←{
          ⎕ML←⎕IO←1
          ⍺←'/\'
          l←1+-⌊/⍺⍳⍨⌽⍵
          (l↓⍵)(l↑⍵)
      }

      Split←{
          ⎕ML←⎕IO←1
          ⍺←⎕UCS 13 10 ⍝ Default is CR+LF
          (⍴,⍺)↓¨⍺{⍵⊂⍨⍺⍷⍵}⍺,⍵
      }

      Last←{
          ⎕ML←⎕IO←1
          ⍺←'.'
          (⍴,⍵)=where←¯1+⌊/⍺⍳⍨⌽,⍵:0⍴⍵
          0 1∊⍨≡r←(-where)↑⍵:r
          ''⍴r
      }

      Nest←{
          ⎕ML←⎕IO←1
          (⊂∘,⍣(0 1∊⍨≡⍵))⍵
      }

    Enlist←{⎕ml←3 ⋄ ∊⍵}

      dmb←{
      ⍝ delete leading, trailing and multiple blanks from scalar or vector.
          ⍺←' '
          ⎕ML←⎕IO←1
          ~0 1∊⍨≡⍵:∇¨⍵
          2=⍴⍴⍵:Mix ∇¨↓⍵
          (,⍺)≡,⍵:''
          w←1↓¯1↓⍺{⍵/⍨~(2⍴⍺)⍷⍵}⍺,⍵,⍺
          (0=⍴⍴⍵)∧1=⍴w:⍬⍴⍵
          w
      }

      dlb←{
      ⍝ Delete leading blanks. Accepts scalar, vector and matrix as argument.
          (2=|≡⍵):∇¨⍵
          (1=⍴⍴⍵):(+/∧\' '=⍵)↓⍵                 ⍝ Vectors (main application)
          (2=⍴⍴⍵):(+/∧\' '=⍵)⌽⍵                 ⍝ Matrix
          (0=⍴⍴⍵):(⎕IO+' '≡⍵)⊃⍵''               ⍝ Scalar
          'Invalid argument'⎕SIGNAL 11
      }

      dtb←{
      ⍝ Delete trailing blanks. Accepts scalar, vector and matrix as argument.
          (2=|≡⍵):∇¨⍵
          (1=⍴⍴⍵):⌽{(+/∧\' '=⍵)↓⍵}⌽⍵            ⍝ Vectors (main application)
          (2=⍴⍴⍵):(-+/∧⌿∧\' '=⌽⍵)↓[2]⍵          ⍝ Matrix
          (0=⍴⍴⍵):(⎕IO+' '≡⍵)⊃⍵''               ⍝ Scalar
          'Invalid argument'⎕SIGNAL 11
      }

    ∇ r←{length}FormatDateTime ts;ts2;formatstring;bool;⎕IO;⎕ML;buffer
⍝ Format the right argument (defaults to ⎕TS if empty) as a string with:
⍝ 'YYYY-MM-DD HH:MM:SS.MILLISECOND
⍝ If the right argument has not 7 but 6 or 3 elements, formatting is done accordingly.
⍝ Via the left argument the length of the right argument can be set to _
⍝ ⍬, 3, 6 or 7; default is 6; ⍬ Accepts any length of the right argument which is 3,6 or 7.
⍝ If the right argument is a simple vector, a string is returned.
⍝ If it is a matrix, a matrix is returned.
      ⎕IO←⎕ML←1
      :If ⍬≡length←{2=⎕NC ⍵:⍎⍵ ⋄ 6}'length'
          length←''⍴¯1↑⍴ts
      :EndIf
      :If ~0∊⍴ts
          :If 2=⍴⍴ts
              buffer←{⍵/⍨0<⊃∘⍴¨⍵~¨⊂' ' 0}↓ts
              'Invalid right argument: must be integer'⎕SIGNAL 11/⍨163≠∪⎕DR¨buffer
              'Invalid right argument: must not be negative'⎕SIGNAL 11/⍨∨/¯1∊¨×¨buffer
              'Invalid right argument: must be simple'⎕SIGNAL 11/⍨1≠∪≡¨buffer
          :Else
              'Invalid right argument: must be integer'⎕SIGNAL 11/⍨163≠⎕DR ts
              'Invalid right argument: must not be negative'⎕SIGNAL 11/⍨¯1∊×,ts
              'Invalid right argument: must be simple'⎕SIGNAL 11/⍨1≠≡ts
          :EndIf
      :EndIf
      :If 2=⍴⍴ts
          ts2←length↑[2]ts
      :Else
          ts2←,[0.5]length↑{0∊⍴⍵:⎕TS ⋄ ts}ts
      :EndIf
      :Select First length
      :Case 3
          formatstring←'ZI4,<->,ZI2,<->,ZI2'
      :CaseList 6 7
          formatstring←'ZI4,<->,ZI2,<->,ZI2,< >,ZI2,<:>,ZI2,<:>,ZI2'
      :Else
          'Invalid left argument'⎕SIGNAL 11
      :EndSelect
      bool←(ts2∨.≠' ')∧ts2∨.≠0
      r←bool⍀formatstring ⎕FMT(6⌊length)↑[2]bool⌿ts2
      :If 7=2⊃⍴ts2
          r←⊃(↓r),¨{0=⍵:'' ⋄ 0∊⍴⍵~' ':'' ⋄ '.',⍕⍵}¨ts2[;7]
      :EndIf
      :If 2≠⍴⍴ts
          r←,r
      :EndIf
    ∇

    ∇ R←CreateUUID;∆UuidCreate;⎕ML;⎕IO
      ⎕ML←⎕IO←1
      '∆UuidCreate'⎕NA'I RPCRT4|UuidCreate >{I1[4] I1[2] I1[2] I1[2] I1[6]}'
      R←2⊃∆UuidCreate 1
      R←'0123456789ABCDEF{-'[1+∊16 17 17 17 17,¨∊¨⍉¨16 16∘⊤¨R],'}'
    ∇

    ∇ r←{flat}ReadUtf8File filename;fno;noOfBytes;bytes;⎕IO;⎕ML;b
    ⍝ This function works also with the Classic version if the file to be read does not contain "real" Unicode data.
    ⍝ By defaults ""ReadUtf8File"" returns a nested vector were each item carries a record.
    ⍝ If you want to get a simple stream specify 'flat' as left argument.
      ⎕IO←⎕ML←1
      r←''
      flat←{0=⎕NC ⍵:0 ⋄ 'flat'≡⍎⍵}'flat'
      :Trap 19 22
          fno←(filename~'"')⎕NTIE 0
      :Else
          ('Could not read file: ',filename)⎕SIGNAL ⎕EN
      :EndTrap
      noOfBytes←⎕NSIZE fno
      bytes←⎕NREAD fno 83,noOfBytes
      ⎕NUNTIE fno
      bytes+←256×bytes<0                 ⍝ Make sure it is unsigned
      bytes↓⍨←3×239 187 191≡3⍴bytes      ⍝ drop a potential UTF-8 marker
      r←'UTF-8'⎕UCS bytes
      :If ~flat
          :If ∨/(⎕UCS 13 10)⍷r
              r←Split r
          :ElseIf ∨/r=⎕UCS 10
              r←(⎕UCS 10)Split r
          :EndIf
      :EndIf
    ∇

    ∇ {r}←{append}WriteUtf8File(file data);fno;fullname;flag;⎕ML;⎕IO;i;max;size;simpleFlag;wasOpenFlag
     ⍝ Write UTF-8 "data" to "file" (WITHOUT a BOM!).
     ⍝ If the left argument is string "append" then "data" is appended _
     ⍝ to an already existing file. If there is no such file yet _
     ⍝ it is created no matter what the left argument is.
     ⍝ In case of "append" the "file" parameter, which is normally _
     ⍝ expected to be a filename, can also be a tie number.
     ⍝ When the tie fails the function tries a couple of times with an
     ⍝ increasing delaybefore giving up.
     ⍝ Note that when a nested vector is passed as right argument then _
     ⍝ CR+LF are added to the end of each item except the very last one. _
     ⍝ When "append" is specified, CR+LF are also added as a prefix to _
     ⍝ the data in case the file does already exist and was not empty yet.
     ⍝ If a simple string is passed it is written as it is: nothing is _
     ⍝ added at all.
      ⎕IO←⎕ML←1
      r←''
      append←{2=⎕NC ⍵:⍎⍵ ⋄ ''}'append'
      'Invalid right argument'⎕SIGNAL 11/⍨~(⊂append)∊'append' 1 0 ''⍬
      append←(⊂append)∊'append' 1
      flag←0
      :If ~simpleFlag←0 1∊⍨≡data
          data←¯2↓⊃,/data,¨⊂⎕UCS 13 10
      :EndIf
      max←5
      i←0
      size←0
      :Repeat
          ⎕DL 1×i>0
          :If wasOpenFlag←append
          :AndIf wasOpenFlag←0=1↑0⍴file
              fno←file
              size←⎕NSIZE fno
              flag←1
          :EndIf
          :If ~wasOpenFlag
              :Trap 19 22
                  fno←(file~'"')⎕NTIE 0 17  ⍝ Open exclusively
                  size←⎕NSIZE fno
                  flag←1
              :Case 22
                  flag←1                    ⍝ That's just fine
              :Else
                  ('Could not open file ',file)⎕SIGNAL ⎕EN
              :EndTrap
          :EndIf
      :Until flag∨max<i←i+1
      :If append
          :If (0<size)∧~simpleFlag
              data,⍨←⎕UCS 13 10
          :EndIf
      :Else
          :Trap 0 ⋄ (file~'"')⎕NERASE fno ⋄ :EndTrap
          fno←(file~'"')⎕NCREATE 0
      :EndIf
      data←⎕UCS'UTF-8'⎕UCS data             ⍝ Enforce UTF-8
      data ⎕NAPPEND fno
      :If ~wasOpenFlag
          ⎕NUNTIE fno
      :EndIf
    ∇

    ∇ {r}←GoToWebPage url;wsh;⎕IO
    ⍝ Fires up the default browser and goes to "url"
    ⍝ For displaying a file rather then a url add "file://".
    ⍝ However, note that "file://" does not work on some systems.
    ⍝ Examples:
    ⍝ GoToWebPage 'file:///c:/my.html'
    ⍝ GoToWebPage 'file://localhost/c:/my.html'
    ⍝ The to calls are equivalent
     
      ⎕WX←1 ⋄ ⎕IO←0 ⋄ ⎕ML←3
      r←⍬
      :If ~∨/'file://'⍷Lowercase url
          url←'http://'{((⍺≢Lowercase(⍴⍺)↑⍵)/⍺),⍵}url
      :EndIf
      'wsh'⎕WC'OLEClient' 'WScript.Shell'
      {}wsh.Run url
    ∇

      FindPathTo←{
      ⍝ Tries to find ⍵ (typically the name of a needed script) in:
      ⍝ 1. the same namespace ⎕THIS is coming from
      ⍝ 2. #
      ⍝ 3. Where it was called from (NOT the same as 1.)
      ⍝ If it fails to find ⍵, an empty string is returned.
          ⎕IO←1 ⋄ ⎕ML←1
          base←{                           ⍝ were are we coming from?
              (~0∊⍴b←{' '~⍨⍵↓⍨-'.'⍳⍨⌽⍵}⍕⎕CLASS ⍵):⍎b
              i←'.'Split⍕⎕THIS             ⍝ Split full path up
              b←∧\~'['∊¨i                  ⍝ "[" = instances - we need something better
              ⍎⊃{⍺,'.',⍵}/b/i              ⍝ That's the base namespace
          }⎕THIS
          0<base.⎕NC ⍵:base                ⍝ Is it in that dir?
          0<#.⎕NC ⍵:#                      ⍝ Is it in root?
          0::''                            ⍝ To be on the save side (may be called from inside an instance)
          path←⍎1⊃⎕NSI~⊂⍕⎕THIS             ⍝ From where got we called?
          0<path.⎕NC ⍵:path                ⍝ May be it's there?
          ''                               ⍝ Give up
      }

    First←{⎕ML←3 ⋄ ↑⍵}                     ⍝ "First" ⎕ml independent
    Mix←{⎕ML←0 ⋄ ↑⍵}                       ⍝ "Mix" ⎕ml independent
    Where←{⎕IO←(⎕IO⊃⎕RSI).⎕IO ⋄ ⍵/⍳⍴,⍵}    ⍝ Return indices for Boolean ⍵; depends on ⎕IO in caller's space
    :EndSection

:EndClass ⍝ WinReg ⍝ WinReg  $Revision: 1068 $ 