﻿:Namespace copyreg ⍝ V3.03
⍝ Copy registry entries to file and optionally between versions
⍝ 2015 05 21 Adam: allow renaming Cmd, code and help improvements
⍝ 2015 05 31 Adam: Auto ⎕PW print
⍝ 2015 06 01 Adam: Auto list columns
⍝ 2015 06 02 Adam: ]?cmd refs all help levels
⍝ 2015 10 14 DanB: Added Test
⍝ 2015 10 21 Adam: .reg now deletes tree before merge (partial export)
⍝ 2015 10 22 Adam: Time stamped files now pad zeros
⍝ 2015 10 23 Adam: Added -folder= and -regonly
⍝ 2015 10 25 Adam: Added documentation for new modifiers
⍝ 2015 10 27 Adam: Changed class name
⍝ 2015 11 12 Adam: moved help layout to framework
⍝ 2016 02 04 Adam: Moved to MSWin Group and fixed bug causing .reg file to go to root folder
⍝ 2016 03 16 Adam: Now copies backup filename to Clipboard
⍝ 2017 03 08 Adam: Filter away phantom installs
⍝ 2017 08 09 Adam: defaults' info in version specification
⍝ 2018 04 18 Adam: ]??cmd → ]cmd -??
⍝ 2018 05 01 Adam: add SVN tag
⍝ 2019 01 16 Adam: help
⍝ 2019 04 15 Adam: handle empty include list
⍝ 2022 01 28 Adam: force user to specify C and 32, and always show them
⍝ 2023 05 30 Adam: Staticise ⎕USING as the Core value works in Framework
⍝ 2023 06 30 Adam: Dynamicise ⎕USING again since the Core value displays a warning
⍝ 2023 08 02 Adam: One more ⎕USING difference for Core vs Framework

    :SECTION MAIN

    ∇ r←List
      →0↓⍨⍴r←⎕NS¨⎕SE.SALTUtils.WIN⍴⊂⍬
      r.Name←⊂'CopyReg'
      r.Group←⊂'MSWin'
      r.Desc←⊂DESC
      r.Parse←⊂'1L -to=  -include=  -exclude=  -folder=  -regonly'
    ∇

    ∇ r←level Help Cmd;h
      r←⊂DESC
      :If 0=level
          r,←⊂'    ]',Cmd,' <sourceinstance> [-to=<destinationinstance>] [-include=<add>] [-exclude=<del>] [-folder=<dir>] [-regonly]'
          r,←⊂''
          r,←⊂']',Cmd,' -?? ⍝ for more information'
      :Else
          r,←'' 'Argument and -to= modifier are one of the following installed APL instances:'
          :If level∊1 3
              r,←⊂'       ',⍕'"'∘,¨,∘'"'¨{⍵,⍨'32 '/⍨~∨/'64 '⍷⍵}¨12↓¨{⍵,' C'/⍨~'U'∊⍵}¨Installs~¨⊂'nicode'
              :If 3=level
                  r,←'' 'Examples:'
                  r,←{⍵,FullVer APLversion ⍵}¨'        '∘,¨13↑¨'13c32' '32u140' '14.1U64'
              :EndIf
          :Else ⍝ 2=level
              r,←⊢⊢⊂'       ',⍕FullVer Installs
              r,←'' '    Instances can be specified in any order and either fully or partially using shorthand:'
              r,←⊢⊢⊂'        ## or ### or ##.#    Version number, for example 14 or 141 or 14.1 (default minor version is .0)'
              r,←⊢⊢⊂'        C or U               Classic/Unicode edition (no default; you must specify edition)'
              r,←⊢⊢⊂'        32 or 64             32/64 bit-width (no default; you must specify bit-width)'
          :EndIf ⍝ Continue 1≤level
          r,←''('    Note:  An instance must be started at least once before using ]',Cmd,'.')
          r,←⊂'    If -to= modifier is omitted, entries will only be copied to a file.'
     
          r,←'' '-include= takes ;-delimited list of keys and values to include, using the following syntaxes:'
          :If level∊1 3
              r,←⊂'        ''''  Name  Path\  Path\Name'
              :If 3=level
                  r,←'' '    If -include= modifier is omitted, the default is:'
                  r,←⊂'       ',⍕(INCLUDEKEYS~(⊂'')),⍨(⊂'''''')/⍨(⊂'')∊INCLUDEKEYS
                  r,←'' 'The default list may be included by adding plus (+) at the beginning, i.e. -include=+Path\Name; etc.'
              :EndIf
          :Else ⍝ Continue 1≤level
              r,←⊂'        ''''           Copy all values in the source root, for example ',InsKey ThisVer
              r,←⊂'        Name         Copy the value Name'
              r,←⊂'        Path\        Copy all keys, sub-keys, and values in and below Path'
              r,←⊂'        Path\Name    Copy just the specific value Name in the key Path'
          :EndIf
     
          r,←'' '-exclude= takes ;-delimited list of values to exclude:'
          :If level∊1 3
              r,←⊂'        Name '
              :If 3=level
                  r,←'' '    If -exclude= modifier is omitted, the default is:'
                  r,←⊂'       ',⍕EXCLUDE
                  r,←⊂''
                  r,←⊂'    The default list may be included by adding plus (+) at the beginning, i.e. -exclude=+Path\Name; etc.'
                  r,←⊂''
                  h←'    Note:  When copying a whole tree (with -include=Path\), '
                  h,←'any value whose data begins with ''//'' or contains the string '':\'' '
                  h,←'or '':/'' will be skipped. This restriction can be circumvented '
                  h,←'by explicitly including value names with -include=.'
                  r,←⊂h
              :EndIf
          :Else ⍝ 2=level
              r,←⊂'        Name         When copying a whole tree (with -include=Path\), skip any values named Name'
          :EndIf
     
          r,←'' '-folder= where to save the .reg file:'
          :If level∊1 3
              r,←⊂'        D:\Path\'
              :If 3=level
                  r,←'' '    If -folder= modifier is omitted, the default for each installed instance is:'
                  r,←'   '∘,¨↓⍕↑{'    '∘,¨(FullVer ⍵)({0::'(unavailable)' ⋄ DocDir ⍵}⍵)}¨Installs
              :EndIf
          :Else ⍝ 2=level
              h←'        D:\Path\     Save registry settings in the Path folder on the D: drive (default is "'
              h,←{0::'(unavailable)' ⋄ DocDir ⍵}ThisVer
              r,←⊂h,'" for this instance)'
          :EndIf
     
          r,←'' '-regonly create a .reg file instead of actually copying the settings'
          :If 3=level
              r,←'' '    To copy the default selection of registry settings (for example to apply to another computer''s installation of this instance):'
              r,←⊂'        ]',Cmd,' ',(12↓ThisVer~'nicode'),' -regonly'
     
          :ElseIf 2=level
              r,←⊂'    This modifier can be used with or without -to=:'
              r,←⊂'        With -to=    The file can be used to make the changes when ready.'
              h←⊢⊢'        Without -to= The file can be used to copy only the selected settings to '
              h,←'installations of the same instance on another computer, or to restore previous settings'
              r,←⊂h
          :EndIf
     
          :If 1≤level
              r,←⊂''
          :EndIf
      :EndIf
      :If 2≠level
          r,←⊂']',Cmd,' -??? ⍝ for details'
      :EndIf
      :If 3≠level
          r,←⊂']',Cmd,' -???? ⍝ for examples and defaults'
      :EndIf
      h←'WARNING:  Inappropriate registry entries can lead to unexpected '
      h,←'and ill-defined behaviour in Dyalog APL. '
      h,←'This utililty saves the affected registry entries to a time-stamped '
      h,←'.reg file before making any changes. Double-click this file to restore any settings '
      h,←'that were changed after the indicated time.'
      r,←''h
    ∇

    ∇ r←Run(Cmd Input);from;to;incl;excl;file;msg;fromtext;totext;nofrom;noto;migrate;ins;reg;dist;regonly;tn;dir;qfileq;cl
      ins←Installs
      fromtext←1⊃Input.Arguments
      totext←''Input.Switch'to'              ⍝ Default is ''
      migrate←''≢totext
      regonly←Input.regonly                  ⍝ Create only a .reg file - don't actually do the migration
      dir←Input.Switch'folder'
      (from to)←APLversion¨fromtext totext   ⍝ Standardize
      :If regonly>migrate ⋄ to←from ⋄ :EndIf ⍝ If dest. wasn't given, but we just want .reg file, assume same ver.
      (nofrom noto)←~from to∊ins             ⍝ Installed?
     
      msg←'": Unable to resolve!'
      ⎕←'Source="',(1+nofrom)⊃(from,'"')(fromtext,msg)
      :If migrate∨regonly
          ⎕←'Destination="',(1+noto>regonly)⊃(to,'"')(totext,msg)
      :Else
          (to noto)←from nofrom
      :EndIf
      ⎕←''
     
      ⍝:If nofrom∨noto∧migrate∨~regonly ⍝ dest. is only req. if we are actually migrating, not only making .reg file
      :If nofrom∨noto∧~regonly∧~migrate∧~''≢dir
          msg←'Must be one of the following installed instances:',CR,∊'    '∘,¨ins,¨CR ⍝ Global from Run
          msg,←'N.B. A version must be run once before using ]',Cmd,'.' ⍝ Global from Run
          msg,←CR,'      ]',Cmd,' -??? ⍝ for info on shorthand version specification like 141U32.'
          msg ⎕SIGNAL 11
     
      :Else  ⍝ Everything is ok - proceed!
          :If 0≡dir
              dir←DocDir to
          :EndIf
          file←File dir
          qfileq←'"',file,'"'
     
          :If ~regonly   ⍝ regonly implies no backup (the output file is only what is to be moved)
              :If file Backup to
                  ⎕←from,' registry entries successfully copied to file.'
                  ⎕←'To revert any changes made from now on, double-click'
                  ⎕←qfileq,'.'
                  'cl'⎕WC'Clipboard' ⋄ 'cl'⎕WS'Text'qfileq
                  ⎕←'This filename has been copied to the Clipboard.',CR
              :Else
                  'Copying to file failed! No changes were made.'⎕SIGNAL 19
              :EndIf
          :EndIf
     
          :If migrate∨regonly
              incl←INCLUDEKEYS Merge''Input.Switch'include'
              excl←EXCLUDE Merge''Input.Switch'exclude'
     
              reg←regonly CopyReg from to incl excl
              :If regonly
                  tn←file ⎕NCREATE 0   ⍝ New file in appropriate folder
                  (reg,⍨⎕UCS 65279)⎕NAPPEND tn,160 ⍝ Prepend BOM which also ensures double-byte
                  ⎕NUNTIE tn
                  r←from,' registry entries successfully copied to file. To setup ',to,',',CR
                  r,←'double-click ',qfileq,'.'
              :Else
                  r←'Registry entries successfully copied from ',from,' to ',to,'.'
              :EndIf
          :Else
              r←'To copy registry entries between instances, use the -to= modifier.'
          :EndIf
      :EndIf
    ∇

    :ENDSECTION

    :SECTION SUBROUTINES

    ∇ reg←REGONLY CopyReg(From To Incl ExclValues);P;Everything;ST;FromPath;pathnames;paths
      ⍝ Copy registry entries between installed APL/W versions
      P←'HKEY_CURRENT_USER\',DYAPATH,'\'
     
      Everything←ReadTree FromPath←P,From
      ST←Everything ShouldTransfer FromPath Incl ExclValues
      ST[;1]←(1+⍴FromPath)↓¨ST[;1]   ⍝ Strip old root path
      pathnames←(P,To,'\')∘,¨ST[;1]  ⍝ Replace with new root path+sub-key path
      ST[;1]←StripPath¨pathnames     ⍝ Values
      paths←(-1+⍴¨ST[;1])↓¨pathnames ⍝ Paths
     
      reg←'Windows Registry Editor Version 5.00',CR,LF,∊paths CopyValue EachOf↓ST  ⍝ Copy them and capture actions
    ∇

    ∇ entry←destinationKey CopyValue sourceValue;kind;name;destSubKey;sourceSubKey;user;obj;t;root;⎕USING;base;n;subKeyName
     ⍝ Copy all the values from source to destination
     ⍝ The 2 arguments are:
     ⍝   (Name Type Data)
     ⍝   Destination                                            object
     ⍝ Ex:                                                        ↓
     ⍝  'HKEY_CURRENT_USER\Software\ABC\Ver2' CopyValue 'name' [Binary] 0
      ⎕USING←Win32
      root←'HKEY_USERS' 'HKEY_CURRENT_CONFIG' 'HKEY_CLASSES_ROOT' 'HKEY_CURRENT_USER' 'HKEY_LOCAL_MACHINE'
      base←Registry.(Users CurrentConfig ClassesRoot CurrentUser LocalMachine)
      t←base[root⍳⊂(¯1+n←t⍳'\')↑t←destinationKey]
      destinationKey←t.CreateSubKey⊂n↓destinationKey
      ⎕USING←'System'
     
      (name kind obj)←sourceValue
      entry←CR,LF,CR,LF,'[',(⍕destinationKey),']',CR,LF,'"',name,'"=' ⍝ [keyname] ⋄ ⋄ "value"=
      :Select t←⍕kind
      :CaseList 'Binary' 'None'
          entry,←'hex:',1↓,',',Hex⍉0 16⊤obj                        ⍝ hex:d4,a1,0g
          obj←Array.CreateInstance Byte,⍴t←{⍵-256×⍵>127}obj        ⍝ create a Byte[] Object
          obj(Byte Cast)t
      :Case 'MultiString'
          entry,←'hex(7):',1↓,',',2 2⍴⍤1⊢Hex⍉(4/16)⊤⎕UCS obj       ⍝ hex(7) with 00,00 as separator ⍝⍝⍝ ??? ASK DAN !!!
          obj←Array.CreateInstance String,⍴t←obj                   ⍝ create a String[] Object
          obj(String Cast)t
      :Else  ⍝ SZ (String)
          entry,←'"','"',⍨('\\' '"'⎕R'\\\\' '\\"')obj              ⍝ in quotes, with \ and " as \\ and \"
      :EndSelect
      :If ~REGONLY ⍝ Global from CopyReg
          destinationKey.SetValue(name obj kind)
      :EndIf
      destinationKey.Close
     
    ∇

    ∇ ntd←ReadTree sourceKey;kind;name;sourceSubKey;user;obj;t;root;⎕USING;base;n;subKeyName;path
     ⍝ Read all the values from source and return a 3 col table of Name,Type,Data
     ⍝ The argument may be a string or a reference to a key
     ⍝ Ex:
     ⍝  ReadTree 'HKEY_CURRENT_USER\Software\ABC\Ver1'
     
      ntd←0 3⍴⍬
      :If ~0=≡sourceKey
          ⎕USING←Win32
          root←'HKEY_USERS' 'HKEY_CURRENT_CONFIG' 'HKEY_CLASSES_ROOT' 'HKEY_CURRENT_USER' 'HKEY_LOCAL_MACHINE'
          base←Registry.(Users CurrentConfig ClassesRoot CurrentUser LocalMachine)
          t←base[root⍳⊂(¯1+n←t⍳'\')↑t←sourceKey]
          sourceKey←t.OpenSubKey⊂n↓sourceKey
      :EndIf
     
      :For name :In sourceKey.GetValueNames
          obj←sourceKey.GetValue⊂name
          kind←sourceKey.GetValueKind⊂name
          path←(⍕sourceKey),'\',name
          ntd⍪←path kind obj
      :EndFor
     ⍝ For Each subKey:
     ⍝ - Call myself on each subKey
      :For subKeyName :In sourceKey.GetSubKeyNames
          sourceSubKey←sourceKey.OpenSubKey⊂subKeyName
          ntd⍪←ReadTree sourceSubKey
          sourceSubKey.Close
      :EndFor
      sourceKey.Close
    ∇

    ∇ st←everything ShouldTransfer(frompath incl exclvalues);b;iv;ik;ev;ed;inclkeys;inclvalues;inclroot;col1
     ⍝ List ShouldTransfer (IncludeList ExcludeList)
     ⍝
     ⍝ Result is ⍺ with lines removed according to these rules:
     ⍝
     ⍝ IncludeList:
     ⍝   ''     Copy all values in the source root (e.g. HKEY_CURRENT_USER\Software\Dyalog\Dyalog APL/W 14.0)
     ⍝   'Name'    Copy the value Name
     ⍝   'Path\'    Copy all keys, sub-keys, and values in and below Path
     ⍝   'Path\Name'    Copy just the specific value Name in the key Path
     ⍝
     ⍝ ExcludeList
     ⍝   'Name'    When copying a whole tree (with Path\), skip any values named Name
     ⍝
     ⍝ N.B.:
     ⍝   When copying a whole tree (with 'Path\'), any value whose data begins with '//' or contains the string ':\' or ':/' will be skipped.
     ⍝   This restriction can be circumvented by explicitly including value names in IncludeList.
     
      iv←0
      inclroot←(⊂'')∊incl
      col1←UCase¨everything[;1]
     
      ed←IsPath everything[;3] ⍝ "excludedata"
      :If inclroot             ⍝ Catch special '' case
          incl~←⊂''
          iv←ed⍱'\'∊¨(1+⍴frompath)↓¨col1
      :EndIf
     
      incl←(frompath,'\')∘,¨incl
      (inclkeys inclvalues)←(b(~b←'\'=∊¯1↑¨incl))/¨⊂incl
     
      iv∨←col1∊UCase¨inclvalues
      ik←⊃∨/{(⊂⍵)≡¨((⍴⍵)↑¨col1)}¨UCase¨inclkeys
      ev←(StripPath¨col1)∊UCase¨exclvalues
     
      st←(iv∨ik∧~ev∨ed)⌿everything
    ∇

    ∇ ok←file Backup ver;⎕USING;task;file;in;out;tmp
    ⍝ Export registry tree of APL version ⍵ to its install folder
      ⎕USING←'System.Diagnostics,System',⊃Core⌽'.dll' '.Diagnostics.Process'
      tmp←'tmp',⍨file
      task←Process.Start'reg'('export ',(InsKey ver),' "',tmp,'" /y')
     
      ok←0 ⍝ No file name on fail
     
     ⍝ Now we add a line to the new .reg file to remove the entire tree before merging
      :If task.WaitForExit 2000 ⍝ wait max 2 seconds
          in←tmp ⎕NTIE 0
          out←file ⎕NCREATE 0
         ⍝ Get the first subkey header, double it, and insert a dash
          out('(\[)([^]]+\])'⎕R'\1-\2\r\n\r\n\0'⍠('Mode' 'D')('ML' 1)('OutEnc' 'UTF16LE-BOM'))in
          tmp ⎕NERASE in
          ⎕NUNTIE out
          ok←1
      :EndIf
     
    ∇

    ∇ BadVersion;msg
      msg←'Must be one of the following installed instances:',CR,∊'    '∘,¨ins,¨CR ⍝ Global from Run
      msg,←'N.B. A version must be run once before using ]',Cmd,'.' ⍝ Global from Run
      msg,←CR,'      ]',Cmd,' -??? ⍝ for info on shorthand version specification like 141U32.'
      msg ⎕SIGNAL 11
    ∇

    :ENDSECTION

    :SECTION UTILS

    FullVer←'W ' '\d$'⎕R'W-32 ' '& Classic'

    EachOf←{0∊⍴⍵:⍵ ⋄ ⍺←⊢ ⋄ ⍺ ⍺⍺¨⍵}

    Cast←{~0∊⍴⍵:⍵∘(⍺⍺{⍵⍵.SetValue⍠'CastToTypes'(⍺⍺ Int32)⊢⍺[⍵],⍵-⎕IO}⍺)¨⍳⍴⍵}

    Merge←{0∊⍴⍵:⍺ ⋄ add←'+'∊1↑⍵ ⋄ names←';',add↓⍵ ⋄ ∪⍺,⍣add⊢1↓¨(names=';')⊂names}

    Hex←{'0123456789abcdef'[1+⍵]}

    IsPath←{(⍳⍴⍵)∊1+('^//|:/|:\\'⎕S 2)⍕¨⍵}

    StripPath←{(1-(⌽⍵)⍳'\')↑⍵}

    Dyalog←{⎕USING←Win32 ⋄ Registry.CurrentUser.OpenSubKey⊂DYAPATH,⍵}

    ∇ s←UCase s;b;i;n ⍝ UpperCase string
      n←⍴LCASE ⋄ →(∨/b←n≥i←LCASE⍳s)↓0
      (b/s)←⎕A[b/i]
    ∇

    ∇ v←APLversion input;nums;bit;vno;ucs;curvno
     ⍝ Returns an APL version specification matching the format used in the registry based on arg
     ⍝ Argument is any combination of the following:
     ⍝ '##' or '##.#' ⍝ Version number, e.g. '14' or '14.1'
     ⍝ 'C' or 'U'     ⍝ Classic/Unicode edition of current or specified version
     ⍝ '32' or '64'   ⍝ 32/64 bit version of current or specified version
     
      input←,⍕input
      input[1↓('.'=input)/⍳⍴input]←' '          ⍝ Remove all but first .
      input←('32|64' '\pL'⎕R' \u& '⍠1)input     ⍝ 32.1 ←→ 32 .1 ⋄ Uppercase and surround all chars with spaces
     
      nums←↑(//)' -/'⎕VFI input
      :If ∨/'UC'∊input     ⍝ remove line in 19.0    ⍝ force specification of edition
          ucs←' Unicode'/⍨'U'=⍬⍴input∩'UC'          ⍝ If a U occurs before the first C
          :If ∨/32 64∊nums ⍝ remove line in 19.0    ⍝ force specification of bit
              bit←'-64'/⍨~32∊nums
          :Else            ⍝ remove line in
              bit←'-00'    ⍝ remove line in 19.0
          :EndIf           ⍝ remove line in 19.0
      :Else                ⍝ remove line in 19.0
          ucs←' X'         ⍝ remove line in 19.0
          bit←'-00'
      :EndIf               ⍝ remove line in 19.0
     
      vno←|⍬⍴nums~32 64                         ⍝ Don't mistake 32 and 64 for ver nums
      vno÷←10*¯1+1⌈⌊10⍟0.1⌈vno                  ⍝ 141 ←→ 14.1
      vno÷←10*33≤vno                            ⍝ 33 ←→ 3.3
      vno←(1,⍨3+×⌊10⍟0.01⌈vno)⍕vno              ⍝ Include exactly one decimal
     
      v←'Dyalog APL/W',bit,' ',vno,ucs
    ∇

    ∇ ts←File dir
    ⍝ Time Stamped .reg file name in specified dir
      dir,←'\'~⊃⌽dir       ⍝ Ensure final \
      ts←'0'∘,¨⍕¨⎕TS       ⍝ Pad 0s
      ts↑¨⍨←-4 2 2 2 2 2 3 ⍝ Chop
      ts←1↓∊'-',⍪ts        ⍝ Insert dashes
      ts,←'.reg'           ⍝ Append extension
      ts,⍨←dir             ⍝ Prepend dir
    ∇

    ∇ ins←DocDir ver
    ⍝ Return installation folder of APL version ⍵
      :Trap 0
          ins←(Dyalog'\',' X|-00'⎕R''⊢APLversion ver).GetValue⊂'log_file'
      :Else
          ⎕DMX.EN ⎕SIGNAL⍨'The registry tree of ',ver,' appears corrupted. Please reinstall.'
      :EndTrap
      ins←(-(⌽ins)⍳'\')↓ins
    ∇

    InsKey←{'"HKEY_CURRENT_USER\',DYAPATH,'\',⍵,'"'}

    :ENDSECTION

    :SECTION CONSTANTS

    ⎕IO←1 ⋄ ⎕ML←1 ⋄ ⎕WX←3

    (CR LF)←⎕UCS 13 10

    DESC←'Copy registry entries to file and, optionally, between Dyalog instances'

    DYAPATH←'Software\Dyalog'    ⍝ This key is hardcoded into exe

    EXCLUDE←'DyalogEmailAddress' 'DyalogWebSite' 'CommandFolder'

    INCLUDEKEYS←'' 'AutoComplete\' 'Colours\' 'Editor\' 'Explorer\' 'files\file_stack_size' 'KeyboardShortcuts\'
    INCLUDEKEYS,←'Printing\' 'SALT\' 'Threads\' 'Search\' 'Captions\' 'WindowRects\'

    LCASE←'abcdefghijklmnopqrstuvwxyz'

    ∇ ins←Installs
    ⍝ Return list of installed Dyalog APL versions
      ins←(Dyalog'').GetSubKeyNames
      ins/⍨←∨/¨(⊂'Dyalog APL/W')⍷¨ins ⍝ remove non-installs
      ins/⍨←(⊂⊂'SALT')∊¨(Dyalog¨'\',¨ins).GetSubKeyNames ⍝ remove phantom installs
    ∇

    ∇ ver←ThisVer
      ver←APLversion ⎕SE.SALTUtils.APLV,⍕'UC'⊃⍨80 82⍳⎕DR''
    ∇

    ∇ r←Win32
      r←'Microsoft.Win32',Core/',Microsoft.Win32.Registry'
    ∇


    ∇ r←Core
      r←1=⊃2250⌶0
    ∇
    :ENDSECTION

    :SECTION TEST

    ∇ r←Test dummy;⎕USING;D;U31;S;t;k;sk;U31id;assert
⍝ This pgm makes sure V31.9U Key exists
      assert←{x←÷⍵}
      ⎕USING←Win32
      D←Registry.CurrentUser.OpenSubKey'Software\Dyalog'RegistryKeyPermissionCheck.ReadWriteSubTree
      {90::D.DeleteSubKeyTree⊂⍵}U31id←'Dyalog APL/W-64 31.9 Unicode'
      {90::D.DeleteSubKey⊂⍵}U31id  ⍝ delete it
      D.Close
⍝ Reopen to create anew
      D←Registry.CurrentUser.OpenSubKey'Software\Dyalog'RegistryKeyPermissionCheck.ReadWriteSubTree
      U31←D.CreateSubKey⊂U31id
      U31.SetValue'log_file' 'xyz'  ⍝ we need this for the cmd - value unimportant
      S←U31.CreateSubKey⊂'SALT'
      S.SetValue'SourceFolder' 'C:\tmp\'
⍝ Test a single value
      ⎕SE.UCMD'copyreg 14.1u -to=31.9u64 -include=aplk'
      assert 0<⍴t←U31.GetValue⊂'aplk'
      assert 2∊⍴t←U31.GetValueNames
⍝ Test 2 values
      ⎕SE.UCMD'copyreg 14.1c64 -to=31.9u64 -include=aplt;aplnid'
      assert 0<⍴t←U31.GetValue⊂'aplnid'
      assert 4∊⍴t←U31.GetValueNames
⍝ Test a single key
      ⎕SE.UCMD'copyreg 14.1u64 -to=31.9u64 -include=',(k←'AutoComplete'),'\'
      sk←U31.OpenSubKey⊂k
      assert 3<⍴t←sk.GetValueNames
⍝ Test a 2 keys with an exception
      ⎕SE.UCMD'copyreg 14.1 u64 -to=31.9u64 -include=Files;',(k←'explorer'),'\ -excl=file_stack_size'
      sk←U31.OpenSubKey⊂k
      assert 2<⍴t←sk.GetValueNames
⍝ Test adding to exceptions
      ⎕SE.UCMD'copyreg 14.1u -to=31.9u64 -include=',(k←'salt'),'\ -excl=+addsalt'
      sk←U31.OpenSubKey⊂k
      assert∧/~'ADDSALT' 'COMMANDFOLDER'∊⎕SE.SALTUtils.uCase¨t←sk.GetValueNames
⍝ Test all
      ⎕SE.UCMD'copyreg 141unicode -to=31.9u64'
      sk←U31.OpenSubKey⊂'Printing'
      assert 20<⍴sk.GetValueNames
      sk←U31.OpenSubKey⊂'colours\schemes'
      assert 12<⍴t←sk.GetSubKeyNames
      U31.Close
      D.DeleteSubKeyTree⊂U31id
      D.Close
    ∇

    :ENDSECTION

:EndNamespace ⍝ copyreg  $Revision$
