﻿:Namespace  JSutils ⍝ V1.18
⍝ John Scholes' utilities
⍝ 2015 12 30 Adam: updated Check help
⍝ 2016 02 17 Adam: fixed bugs in Dinput help
⍝ 2016 03 11 DanB: changed group of Calendar back to TOOLS
⍝ 2017 05 16 JohnS: Typo in ]?Peek
⍝ 2017 05 26 Adam: typos in ]??calendar and ]??dinput
⍝ 2017 06 17 Adam: Dinput → DInput
⍝ 2017 09 27 Adam: [14952] ]map does ⎕← if result isn't used.
⍝ 2018 02 13 Adam: Rename ]Map → ]Tree
⍝ 2018 02 13 Adam: Rename ]Tree → ]Map
⍝ 2018 04 18 Adam: ]??cmd → ]cmd -??
⍝ 2018 05 08 Adam: help tweaks
⍝ 2018 06 16 JohnS: ]map accepts ref-returning expression as argument
⍝ 2019 01 29 Adam: Help

    ⎕IO←1 ⋄ ⎕ML←1 ⋄ CR←⎕ucs 13

    AllCmds←'Calendar' 'Peek' 'DInput' 'Map' 'Check'

    ∇ r←List;desc;namesr
      :Access Shared Public
      r←{⎕NS ⍬}¨names←AllCmds
      desc←{r[⍺].Desc←⍵}  ⍝ short description.
      r.Name←names
      r.Group←'TOOLS' 'WS' 'FN' 'WS' 'WS'
      'calendar'{}1 desc'Display calendar'
      'wspeek'{}2 desc'Execute expression in temporary copy of workspace'
      'dinput'{}3 desc'Define a multi-line dfn, dop or derived function/operator, or execute a multi-line expression'
      'map'{}4 desc'Display namespace treeview'
      'check'{}5 desc'Perform workspace integrity check'
      r.Parse←'' '' '' '' '-integrity'     ⍝ All fns take raw input buffer as arg except Check.
    ∇

    ∇ r←lev Help Cmd;i;h;s
      :Access Shared Public
      i←AllCmds⍳⊂Cmd
      r←List.Desc[i]
      :Select i
      :Case 1⊣'calendar'
          r,←⊂'    ]',Cmd,' [<month>] [<year>]'
          :If 0<lev
              r,←⊂'Argument is month (name or number) and/or year (default is current month)'
              r,←'' 'Examples:'
              r,←⊂'      ]',Cmd,'           ⍝ current month'
              r,←⊂'      ]',Cmd,' April     ⍝ specific month this year'
              r,←⊂'      ]',Cmd,' sept 1752 ⍝ specific month and year'
              r,←⊂'      ]',Cmd,' dec 78    ⍝ December of closest ''78 to now'
              r,←⊂'      ]',Cmd,' 1 76      ⍝ January ''76'
              r,←⊂'      ]',Cmd,' 2012      ⍝ calendar for whole year'
          :EndIf
      :Case 2⊣'peek'
          r,←⊂'    ]',Cmd,' <ws> [<expression>|→]'
          :If lev>0
              r,←⊂'<ws>          workspace to be copied temporarily. "." means the current workspace.'
              r,←⊂'<expression>  expression to execute. WARNING: defaults to "⍎⎕LX" which can include calling ⎕OFF, terminating the APL session.'
                h←'→             start simple interactive ⍎⍞ loop, indicated by four leading dots "····". '
              h,←'The loop is terminated on input of a single "→" character'
              r,←h''
              r,←⊂'Examples:'
              r,←⊂'          ]',Cmd,' dfns  0 disp queens 5    ⍝ exec expr in copy of dfns'
              r,←⊂'          ]',Cmd,' dfns  tree ⎕this         ⍝ ⎕this is root space of copy'
              r,←⊂'          ]',Cmd,' tube  paris.⎕ed''source''  ⍝ use editor to view variable in copy of ws'
              r,←⊂'          ]',Cmd,' dfns  cols ⎕nl 3 4       ⍝ list fns and ops in dfns.dws'
              r,←⊂'          ]',Cmd,' . myfn                   ⍝ saved version of myfn'
              r,←⊂'          ]',Cmd,' dfns →                   ⍝ interactive loop in copied workspace'
              r,←⊂'    ····fns ops ← ⎕nl¨ -3 4  ⍝ vars for duration of loop'
              r,←⊂'    ····⍴fns                 ⍝ simple expression in copy'
              r,←⊂'    178'
              r,←⊂'    ····lines ← ⊃∘⍴∘⎕cr      ⍝ temporary function definition'
              r,←⊂'    ····+/lines¨fns,ops      ⍝ using all of the above'
              r,←⊂'    6014'
              r,←⊂'    ····→                    ⍝ termination of loop'
          :EndIf
      :Case 3⊣'dinput'
          r,←⊂'    ]',Cmd,' [<name>]'
          :If 0<lev
              h←'<name> is the function or operator to be defined. If none is specified, then '
              h,←'the multi-line expression is evaluated upon closure of the final curly brace. '
              h,←'During input accumulation, nesting level is indented and indicated with dots "·   ·", '
              h,←'which will be replaced by space characters. To abort the loop, enter a single "→" character. '
              r,←h''
              h←'You can amend a previously submitted multi-line input simply by '
              h,←'making corrections in the session history, making sure that all '
              h,←'lines, including the ]dinput ..., are marked for resubmission.'
              r,←h''
              r,←'Examples:' '' 'Entering and executing a multi-line expression:'
              r,←⊂'          ]',Cmd
              r,←⊂'    ·   {            ⍝ A thre-line monadic dfn to duplicate its argument, like {⍵ ⍵}'
              r,←⊂'    ·   ·   ⍵ ⍵'
              r,←⊂'    ·   }{           ⍝ A three-line monadic dop to apply its operand twice on the argument, like {⍺⍺ ⍺⍺ ⍵}'
              r,←⊂'    ·   ·   ⍺⍺ ⍺⍺ ⍵'
              r,←⊂'    ·   }7           ⍝ The argument to the derived function'
              r,←⊂'    ┌→──┬───┐'
              r,←⊂'    │7 7│7 7│'
              r,←⊂'    └~─→┴~─→┘'
              r,←'' 'Defining a multi-line dfn:'
              r,←⊂'          ]',Cmd,'  gcd     ⍝ Multi-line definition'
              r,←⊂'    ·   gcd←{           ⍝ (c) Euclid'
              r,←⊂'    ·   ·   ⍺=0:|⍵'
              r,←⊂'    ·   ·   ⍵ ∇ |⍺-⍵'
              r,←⊂'    ·   }'
              r,←⊂''
          :EndIf
      :Case 4⊣'map'
          r,←⊂'    ]',Cmd,' [<namespace>]'
          r,←⊂''
          :If 0=lev
              r,←⊂']',Cmd,' -?? ⍝ for more information and examples'
          :Else
              h←'<namespace>  where (default: current namespace) to map '
              h,←'variables, functions, and operators. Each row has items of one class, '
              h,←'identified by a bullet: ~ for variables, ∇ for functions, and ∘ for '
              h,←'operators. Nested namespaces are displayed recursively.'
              r,←h'Examples:'
              r,←⊂'        ]',Cmd,'             ⍝ map of current namespace'
              r,←⊂'        ]',Cmd,' ⎕SE.Dyalog  ⍝ map of Dyalog namespace'
              ⎕SE.UCMD' +\]'⎕R's←'⊃⌽r
              r,←'^'⎕R'    '⍠'ResultText' 'Nested'⊢s
              :If 0
                  r,←⊂'⎕SE.Dyalog'
                  r,←⊂'·   Callbacks'
                  r,←⊂'·   ·   ~ ConfirmEdit'
                  r,←⊂'·   ·   ∇ WSLoaded'
                  r,←⊂'·   Out'
                  r,←⊂'·   ·   ∇ Filter Rows SD'
                  r,←⊂'·   ·   ∘ Box'
                  r,←⊂'·   ·   B'
                  r,←⊂'·   ·   ·   ~ chars fns state style'
                  r,←⊂'·   ·   R'
                  r,←⊂'·   ·   ·   ~ dots fns fold state'
                  r,←⊂'·   Utils'
                  r,←⊂'·   ·   ~ Version lc uc'
                  r,←⊂'·   ·   ∇ cut disp display dmb drvSrc dtb fromXML fromto lcase repObj'
                  r,←⊂'·   ·   ∇ showCol showRow toMatrix toVector toXML trimEnds txtreplace'
                  r,←⊂'·   ·   ∇ ucase where'
                  r,←⊂'·   ·   SALT_Data → ⎕SE.[Namespace]'
              :EndIf
          :EndIf
          r,←'' 'NOTE:  Uses the "tree" function from the supplied "dfns.dws" workspace. For details:'
          r,←⊂'      ]peek dfns.dws ⎕ed''notes.tree'''
          :Return
      :Case 5⊣'check'
          r,←⊂'    ]',Cmd,' [-integrity]'
          :If 0<lev
              r,←'Compare the workspace and scripted namespaces to find items that occur in one and not the other. ' ''
              r,←⊂'-integrity  also check integrity of low-level workspace representation in memory. WARNING: This can trigger a system error if a serious problem is found.'
              r,←'' 'Example:' '        ⎕FIX '':namespace X'' ''a←⍳9'' '':endnamespace'''
              r,←⊂'        X.Foo←{123}'
              r,←⊂'        X.abc←90'
              r,←⊂'        ]',Cmd
              r,←⊂'    These extra objects are found in scripted #.X: Foo  abc'
          :EndIf
      :EndSelect
      :If 0=lev
          r,←⊂''
          r,←⊂']',Cmd,' -?? ⍝ for more information and examples'
      :Else
          r←(2↑r),(⊂''),2↓r
      :EndIf
    ∇

    ∇ r←checkws A
     ⍝ Perform some ws verifications
     ⍝ Check for names in namespaces that do not match the script
      :If A.integrity ⋄ 2 ⎕NQ'.' 'wscheck' ⋄ :EndIf
      r←∊extranamesinns¨⎕THIS~⍨allnss #
      r,←(0∊⍴r)/'No problem found.'
    ∇

    ∇ ∆_r←extranamesinns ∆_ns;∆_;∆_new;∆_src
     ⍝ Check the names inside a namespace
      →(⍴∆_src←{0::'' ⋄ ⎕SRC ⍵}∆_ns)↓⍴∆_r←⍬
      ∆_new←(⎕NS'').⎕FIX ∆_src
      →(⍴∆_←(∆_ns.⎕NL ∆_src)~(⊂'SALT_Data'),∆_new.⎕NL ∆_src←¯2 3 4 9.1 9.4)↓0
      →(⍴∆_←('∆_'∘≢¨2↑¨∆_)/∆_)↓0
      ∆_r←CR,⍨'These extra objects are found in scripted ',(⍕∆_ns),':',⍕∆_
    ∇

    ∇ nss←allnss from;fnss;nam;len
    ⍝ Return all namespaces starting from 'from'
      nss←⍬
      →(⍴fnss←from.⎕NL ¯9.1)↓0
      fnss←⍕¨nss←from⍎⍕fnss ⋄ len←⍴nam←(⍕from),'.'
      nss←(nam∘≡¨len↑¨fnss)/nss ⍝ eliminate non legitimate namespaces
      →(⍴nss)↓0
      nss,←∊allnss¨nss
    ∇

    ∇ r←Run(Cmd Args)
      :Access Shared Public
      :Select AllCmds⍳⊂Cmd
      :Case 1⊣'calendar' ⋄ r←calendar Args
      :Case 2⊣'peek' ⋄ r←wspeek Args
      :Case 3⊣'dinput' ⋄ r←##.THIS dinput Args
      :Case 4⊣'map'
          r←##.THIS map Args
          :If ~##.RIU
              r←⍬⊤⍬⊣⎕←r
          :EndIf
      :Case 5⊣'check' ⋄ r←checkws Args
      :EndSelect
    ∇

      calendar←{                                      ⍝ Calendar for month or year.
     
          usage←'Usage: ]Calendar [month] [year]'     ⍝ usage
     
          month←{                                     ⍝ calendar for month.
              yyyy mm←⍵                               ⍝ year and month number.
              day←days yyyy mm 1                      ⍝ epoch day for 1st of month.
              mms dds←2↑1↓↓⍉date day+¯1+⍳31           ⍝ 31 month and day numbers.
              fmts←2 0∘⍕¨(mm=mms)/dds                 ⍝ char-formatted day numbers.
              dow←7|day                               ⍝ day of week of 1st of month.
              pad←dow↑0↑fmts                          ⍝ start of month padding.
              dmat←↑{⍺,' ',⍵}/dys⍪6 7⍴42↑pad,fmts     ⍝ day matrix.
              head←(mm⊃months),⍺/' ',yfmt yyyy        ⍝ month [year] header.
              ↑(⊂cntr ¯20↑head),↓{(∨/⍵≠' ')⌿⍵}dmat    ⍝ calendar for month ⍵.
          }
     
          days←{                                      ⍝ days since 1899-12-31 (Meeus).
              ⍺←17520902                              ⍝ start of Gregorian calendar.
              yy mm dd h m s ms←7↑⊂[⍳¯1+⍴⍴⍵]⍵         ⍝ ⎕ts-style 7-item date-time.
              D←dd+(0 60 60 1000⊥↑h m s ms)÷86400000  ⍝ day with fractional part.
              Y M←yy mm+¯1 12×⊂mm≤2                   ⍝ Jan, Feb → month 13 14.
              A←⌊Y÷100                                ⍝ century number.
              B←(⍺<0 100 100⊥↑yy mm dd)×(2-A)+⌊A÷4    ⍝ Gregorian calendar correction.
              ¯2416544+D+B+⊃+/⌊365.25 30.6×Y M+4716 1 ⍝ (fractional) days.
          }
     
          date←{⎕ML←0                                 ⍝ ⎕TS from day number (Meeus).
              ⍺←¯53799                                ⍝ UK Gregorian calendar starts.
              qr←{⊂[1+⍳⍴⍴⍵](0,⍺)⊤⍵}                   ⍝ quotient and remainder.
              Z F←1 qr ⍵+2415020
              a←⌊(Z-1867216.25)÷36524.25
              A←Z+(Z≥⍺+2415021)×1+a-⌊a÷4
              B←A+1524
              C←⌊(B-122.1)÷365.25
              D←⌊C×365.25
              E←⌊(B-D)÷30.6001
              dd df←1 qr(B-D)+F-⌊30.6001×E
              mm←E-1+12×E≥14
              yyyy←C-4715+mm>2
              part←60 60 1000 qr⌊0.5+df×86400000
              ↑[⎕IO-0.5]yyyy mm dd,part
          }
     
          yyyy_mm←{                                   ⍝ month and year.
              2=⍴⍵:(digs¨⍵)infer num¨⍵                ⍝ explicit year and month.
              0=⍴⍵:2↑⎕TS                              ⍝ this year, this month.
              1≠⍴⍵:0                                  ⍝ bad arg vector.
              digs⊃⍵:(num⊃⍵),⊂⍬                       ⍝ year: all months.
              (⊃⎕TS),num⊃⍵                            ⍝ month: month in this year.
          }
     
          infer←{                                     ⍝ guess at month/year combo.
              1=+/⍺:(⍺⍳0)⌽⍵                           ⍝ 2000 feb → 2000 2
              0=+/⍺:0                                 ⍝ june july: bad.
              mm←⍵≤12                                 ⍝ 1..12 interpreted as year.
              1=+/mm:(mm⍳1)⌽⍵                         ⍝ yyyy mm.
              0=+/mm:0                                ⍝ 1914 1918: bad
              2000 0+⌽⍵                               ⍝ 3 4 → 2004 3
          }
     
          y2k←{                                       ⍝ 50-year window around yy.
              ⍵≥100:⍵                                 ⍝ full year yyyy?
              now←100|⊃⎕TS                            ⍝ this year.
              ⍵+1900+100×50>|now-⍵                    ⍝ 98→1998  12→1012
          }
     
          num←{                                       ⍝ number from char vector.
              digs ⍵:⍎⍵                               ⍝ '1984' → 1984
              mms←(⍴⍵)↑¨mons                          ⍝ jan feb ...
              mm←mms⍳⊂lcase ⍵                         ⍝ feb → 2
              mm×mm≤12                                ⍝ fev: bad month.
          }
     
          cntr←{(⌈0.5×+/∧\' '=⍵)⌽⍵}                   ⍝ centred text.
          join←{⍉↑(↓⍉⍺),'   ',↓⍉⍵}                    ⍝ month joiner.
          digs←{∧/⍵∊⎕D}                               ⍝ digit vector.
          yfmt←{(-4⌈⍴⍕⍵)↑'000',⍕⍵}                    ⍝ leading 0s for years <1000.
     
          dys←'Su' 'Mo' 'Tu' 'We' 'Th' 'Fr' 'Sa'      ⍝ day-of-week column headers.
          Q0←'January' 'February' 'March    '~¨' '    ⍝ 1st quarter month names.
          Q1←'April  ' 'May     ' 'June     '~¨' '    ⍝ 2nd   ..      ..    ..
          Q2←'July   ' 'August  ' 'September'~¨' '    ⍝ 3rd   ..      ..    ..
          Q3←'October' 'November' 'December '~¨' '    ⍝ 4th   ..      ..    ..
          months←Q0,Q1,Q2,Q3                          ⍝ month names.
          mons←lcase¨months                           ⍝ for case-free comparison.
     
          wds←list rmcm ⍵                             ⍝ blank-separated words.
          2<⍴wds:usage                                ⍝ too many args: give up.
     
          yyyy mm←⌽y2k\⌽yyyy_mm wds                   ⍝ year and month number.
          0∊yyyy mm:usage                             ⍝ bad date.
          ~mm≡⍬:1 month yyyy mm                       ⍝ calendar for single month.
          year←4 3⍴{0 month yyyy ⍵}¨⍳12               ⍝ each month for given year.
          head←cntr ¯66↑yfmt yyyy                     ⍝ centred year header.
          head⍪,[⍳2]↑join/year                        ⍝ calendar for year ⍵.
      }

      wspeek←{⎕IO←0                               ⍝ Execute expression in temp copy of WS
     
          usage←'Usage: ]Peek wsid [expr ...]'  ⍝ usage.
     
          wsid tail←↑{                            ⍝ wsid and remainder.
              ((⍺≡,'.')⊃⍺ ⎕WSID)⍵                 ⍝ '.' means ⎕wsid.
          }/{                                     ⍝ first word and remainder.
              raw←rmqt ⍵                          ⍝ ignoring quoted '"' chars.
              ~'"'∊raw:vect\head list rmcm ⍵      ⍝ wsid is first word.
              1↓¨('"'=⊃↓↑raw ⍵)⊂⍵                 ⍝ "-delimited file name.
          }rmcm ⍵
     
          wsid∧.=' ':usage                        ⍝ ]ws
          expr←(tail∧.=' ')⊃tail'⍎⎕lx'           ⍝ ]ws wsid → ]ws wsid ⍎⎕lx
                                                ⍝ ]ws wsid expr ...
          ⎕LX ⎕WSID←⊂''                           ⍝ localise session vars.
          nsv1←'⎕ct' '⎕io' '⎕div' '⎕ml'           ⍝ namespace extent ...
          nsv2←'⎕pp' '⎕rl' '⎕rtl' '⎕wx'           ⍝ ... system variables.
          nsv3←'⎕lx' '⎕wsid' '⎕trap'              ⍝ ws extent vars.
     
          copy←'⎕cy''',wsid,''' ⋄ '               ⍝ copy command.
          lasv←'(##.(↑nsv1,nsv2,nsv3))'           ⍝ left argt for sysvar copy.
          11::'Can''t copy "',wsid,'"'            ⍝ copy failed.
          ref←(⎕NS'')⍎copy,lasv,copy,'⎕this'      ⍝ ref to copied ws.
          0::(⊃⎕DM)⎕SIGNAL ⎕EN                    ⍝ pass back all errors.
          6::0 0⍴0                                ⍝ ignore value error.
          _←ref.⎕DF']WSpeek ',wsid,' #'           ⍝ display form, for ]ws wsid ⎕this
          ~expr≡,'→':{⍵}ref⍎expr                  ⍝ single expr: evaluate in copy of ws.
          trim←{('·'≠⍵){⍺\⍺/⍵}⍵}                  ⍝ replace white dots with blanks.
          {                                       ⍝ ⍎⍞ loop until →.
              0::∇ ⎕←lcase(⊃⎕DM)~'⍎'              ⍝ error:: report and continue.
              expr←trim{⍞}⍞←4/'·'                 ⍝ next expr
              ' '∧.=_xp←trim rmcm expr:∇ ⍵        ⍝ ignore null expr.
              _xp≡,'→':⊤⍨⍬                        ⍝ → means "exit"
              rslt←ref⍎expr                       ⍝ evaluate expr in copy of ws.
              '←'∊rmcm rmqt expr:∇ ⍵              ⍝ assignment: continue.
              ∇ ⎕←rslt                            ⍝ display rslt and continue.
          }⍬
      }


      dinput←{⎕IO ⎕ML←0 1                     ⍝ Input of multi-line D-expression.
          ⍺.{⎕FX ⍵}{                          ⍝ fix in calling space.
     
              accm←{                          ⍝ accumulation of lines.
                  0=level sig ⍵:⍺ ⍵           ⍝ outer level: finished.
                  next←get prmt sig ⍵         ⍝ next input line.
                  (,'→')≡next~' ':⍎'→'        ⍝ →: abort
                  defn←⍺,⊂next                ⍝ accumulated definition.
                  expr←⍵ join cm zap next     ⍝ accumulated expression.
                  defn ∇ expr                 ⍝ ... more lines.
              }                               ⍝ :: (defn expr)←defn ∇ expr
     
              prmt←{                          ⍝ '·   '-indented prompt.
                  cont←null or more ⍵         ⍝ continue at this level.
                  tail←cont⊃'}'tab            ⍝ close or continuation
                  (∊(level ⍵)⍴⊂tab),tail      ⍝ indented prompt.
              }
     
              ask←{⍞⊣⍞←⍵}                     ⍝ ⍵-prompted input.
              or←{(⍺⍺ ⍵)∨⍵⍵ ⍵}                ⍝ this or that.
              zap←{(~⍺⍺ ⍵){⍺\⍺/⍵}⍵}           ⍝ blank ⍵ where ⍺.
              level←{⊃⌽depth ⍵,' '}           ⍝ current nesting level.
              depth←{⍵⌊¯1⌽⍵}∘{-⌿+\'{}'∘.=⍵}   ⍝ │·{a{'⍝'}c}⍝│ → 0 0 1 1 2 2 2 1 1 0 0
              qt←{⍵∧¯1⌽⍵}∘{≠\⍵=''''}          ⍝ │·{a{'⍝'}c}⍝│ → 0 0 0 0 0 1 0 0 0 0 0
              cm←∨\∘~∘('⍝'∘≠or qt)            ⍝ │·{a{'⍝'}c}⍝│ → 0 0 0 0 0 0 0 0 0 0 1
              dt←~∘('·'∘≠or cm or qt)         ⍝ │·{a{'⍝'}c}⍝│ → 1 0 0 0 0 0 0 0 0 0 0
              sig←cm or qt zap                ⍝ │ {a{'⍝'}c}⍝│ → │ {a{' '}c}│
              edge←{'{}'∨.=2↑¯1⌽(⍵,⍺)~' '}    ⍝ ...{ or }...
              join←{⍺,(⍺ edge ⍵)↓'⋄',⍵}       ⍝ ⋄-separated segments.
              get←dt zap∘ask                  ⍝ input with blanked leading '·'s.
              lines←{⍵≠⊃⌽⍵}∘depth zap         ⍝ lines in current nested fn.
              sepr←{⍺{1↓¨(⍺=⍵)⊂⍵}⍺,⍵}         ⍝ ⍺-separated lines.
              segs←'⋄'∘sepr∘lines             ⍝ ⋄-separated segments.
              more←{1∊'←:'∊⊃⌽segs ⍵}          ⍝ last line was assign or guard.
              null←{(⊃⌽'⋄',⍵~' ')∊'{⋄'}       ⍝ ignoring null line.
              align←{(4⌊+/∧\' '=⍵)↓⍵}         ⍝ defn lines without indentation.
              tab←'·   '                      ⍝ tab with white dot.
     
              init←(cm zap ⍵)~' '             ⍝ defn name or null.
              asgn←(×⍴init)/'←{'              ⍝ name←{
              head←get tab,init,asgn          ⍝ initial line.
              ixpr←cm zap head                ⍝ initial expression.
              defn expr←(,⊂head)accm ixpr     ⍝ following lines.
              name←⍺⍺ align¨defn              ⍝ external fix of function.
              ' '=⊃0⍴name:0 0⍴0               ⍝ success: finished.
              ⍵⍵ expr                         ⍝ external exec of expression.
     
          }⍺.{                                ⍝ execute in calling space.
              0::⎕EM ⎕EN                      ⍝ catch all errors.
              85::0 0⍴0                       ⍝ no result from assign.
              2::1(85⌶)⍵                      ⍝ catch pre V14.1 version
              85⌶⍵                            ⍝ execute expr.
          }⍵                                  ⍝ initial command arg.
      }

      map←{                             ⍝ namespace map
          0::'Usage: ]Map [namespace]'  ⍝ catch-all
          0=≢⍵~' ':tree ##.THIS         ⍝ map of current space.
          tree ⍺.⍎⍵                    ⍝ map of given space.
      }

      tree←{⎕IO ⎕ML←0 1                           ⍝ Display of namespace tree.
          0 1000::⎕SIGNAL ⎕EN                     ⍝ pass error to caller.
          ⍺←⎕PW ⋄ ⍺{                              ⍝ default print width,
              ⍺⍎⍵                                 ⍝ external execute.
          }{
              exec←⍺⍺                             ⍝ external execute.
     
              trav←{                              ⍝ traversal of namespace tree.
                  here←⍺ tab ⍵,type ⍺             ⍝ current space name.
                  this←here lf∊⍺∘disp¨2 3 4       ⍝ content: vars, fns, ops.
                  subs←cvex ⍺ xnl 9               ⍝ sub-spaces of this space.
                  0=⍴subs:this                    ⍝ no sub-spaces: finished.
                  refs←⍺∘exec¨subs                ⍝ sub-space refs.
                  this,∊refs(⍺ dosub ∇)¨subs      ⍝ process sub-spaces.
              }                                   ⍝ :: [char] ← ref ∇ name
     
              disp←{                              ⍝ Space namelist per class.
                  names←' ',¨cvex ⍺ xnl ⍵         ⍝ namelist for this class.
                  tag←⍵⊃'··~∇∘'                   ⍝ class identifier.
                  (⍺ xtab tag)wrap names          ⍝ wrapped output of namelist.
              }
     
              wrap←⍺{                             ⍝ ⎕pw-wrapping:
                  0=⍴⍵:''                         ⍝ nothing to output: finished.
                  split←+/(1+⍺⍺-⍴⍺)>+\∊⍴¨⍵        ⍝ names that fit this time.
                  here←⍺,∊split↑⍵                 ⍝ tabbed first line, and
                  here lf ⍺ ∇ split↓⍵             ⍝ tabbed remaining ones.
              }
     
              type←{                              ⍝ Namespace tag:
                  id←⍵                            ⍝ name for following ⎕WG:
                  tag←'id'⎕WG'Type'               ⍝ type of namespace.
                  set←↑'Name' 'Root' 'Sess'       ⍝ ignore Self-Evident Type.
                  (~∨/set∧.=4↑tag)/' [',tag,']'   ⍝ [type].¯    ¯       ¯
              }
     
              dosub←{                             ⍝ Check ref immediate child.
                  (⍺≠⍺⍺)∧⍺⍺=⍺.##:⍺ ⍵⍵ ⍵           ⍝ ⍺ is child of ⍺⍺: proceed.
                  fmt←⍕⍺                          ⍝ system display form.
                  df←(1=⍴⍴fmt)⊃'[Namespace]'fmt   ⍝ display form.
                  (⍺⍺ xtab ⍵,' → ',df)lf''        ⍝ show ref to remote space.
              }
     
              depth←{                             ⍝ space depth.
                  df←⍵.⎕DF ⎕NULL                  ⍝ reset display form to default.
                  dots←+/'.'=⍕⍵                   ⍝ dots in fmt => depth of space.
                  {dots}⍵.⎕DF df                  ⍝ reset display form.
              }
     
              tab←(depth ⍵){(∊⍺⍺↓tabs ⍺),⍵}       ⍝ ⍺ space indented ⍵.
              tabs←{(depth ⍵)⍴⊂4↑'·'}             ⍝ tabs: ·   ·   ·
              xtab←{(⍺.⎕NS'')tab ⍵}               ⍝ extra tab.
              lf←{⍺,(2⊃⎕TC),⍵}                    ⍝ separated with <LF> char.
              xnl←{⍺ exec'⎕NL ',⍕⍵}               ⍝ external namelist.
              cvex←~∘' '¨∘↓                       ⍝ char vectors from matrix.
     
              ¯1↓⍵ trav⍕⍵                         ⍝ traversal of space graph.
          }⍵
      }

    COM←'⍝'                                 ⍝ prob with embedded '⍝' in dfn.
    lc←'abcdefghijklmnopqrstuvwxyz'         ⍝ lower case alphabet
    uc←'ABCDEFGHIJKLMNOPQRSTUVWXYZ'         ⍝ upper case alphabet
    lcase←{(lc,⎕AV)[(uc,⎕AV)⍳⍵]}            ⍝ lower-casification,
    rmcm←{(∧\~(COM=⍵)∧qmask ⍵)/⍵}           ⍝ ignoring unquoted comment.
    qmask←{{⍵∨¯1⌽⍵}~≠\⍵=''''}               ⍝ mask of unquoted chars.
    head←{(⊃⍵)(1↓⍵)}                        ⍝ head and tail of list.
    rmqt←{(qmask ⍵){⍺\⍺/⍵}⍵}                ⍝ with blanks for quoted chars.
    list←{(~∘' '¨(1,' '=⍵)⊂' ',⍵)~⊂''}      ⍝ blank-separated word list.
    vect←{¯2↓↑{⍺,' ',⍵}/⍵,'' ''}            ⍝ enlist of blank-separated words.

    :SECTION Test

    ∇ r←Test dummy;T;cmd;n;fn;ns;grps;a
      'a.b.c.d'⎕NS''
      {}÷' #.JSutils.a.b  #.JSutils.a.b.c  #.JSutils.a.b.c.d '≡ns←⍕allnss a
      grps←List.Group
      r←⍬ ⋄ T←{⎕SE.UCMD(⊃grps[AllCmds⍳⊂cmd]),'.',cmd,' ',⍵}
      :For cmd :In AllCmds
          :Trap n←0
              :Select cmd ⍝ Note: The order of the cases is important
              :Case 'Calendar'
                  n←(T'sept 1752')≡5 20⍴'   September 1752   Su Mo Tu We Th Fr Sa       1  2 14 15 1617 18 19 20 21 22 2324 25 26 27 28 29 30'
              :Case 'Peek'
                  n←n∧~T'. n←0'⊣n←1
              :Case 'DInput'
                  {1 ⎕NQ'⎕se' 'KeyPress',⍵}¨'}',⊂⊂'ER'
                  T fn←' Please_ignore_this_strange_line'
                  n←(⎕NR fn)≡,⊂fn,'←{}'
              :Case 'Map'
                  'ns'⎕NS fn
                  n←(T'ns')≡(⍕⎕THIS),'.ns',(⎕UCS 13 183),'   ∇',fn
              :Case 'Check'
                  ⎕EX'fn' 'ns'
                  n←'No problem found.'≡T'-i'
              :EndSelect
          :EndTrap
          r,←n
      :EndFor
    ∇

    :ENDSECTION

:EndNamespace
 ⍝ JSutils  $Revision: 1574 $
