﻿:Namespace split ⍝ V1.09
⍝ Split a file or gather its pieces
⍝ 2015 05 22 Adam: NS header
⍝ 2015 10 27 DanB: modified split
⍝ 2015 11 02 DanB: modified collect
⍝ 2015 11 10 DanB: added compression
⍝ 2017 05 28 Adam: Desc casing
⍝ 2017 07 13 Adam: help tweaks
⍝ 2018 04 18 Adam: ]??cmd → ]cmd -??
⍝ 2019 02 01 Adam: Help update
⍝ 2019 08 16 Adam: Avoid 0E0 notation in info file

    ⎕IO←1 ⋄ ⎕ML←1 ⋄ AllCmds← 'Split' 'Collect' ⋄ (CR LF)←⎕UCS 13 10 ⋄ NL←⌽LF,⎕SE.SALTUtils.WIN/CR


    ∇ r←List
      r←⎕NS¨0/¨AllCmds
   ⍝ Name, group, short description and parsing rules
      r.Name←AllCmds
      r.Group←⊂'File'
      r[1].Desc←'Split a single file into (up to 999) smaller files'
      r[2].Desc←'Merge all the files with the specified path/name{-nnn} into a single file'
      r.Parse←'1 -chunk= -compress -quiet -track[=]' '1 -decompress -newname= -rewrite -quiet'
    ∇

    ∇ r←Run(Cmd Input)
      :Select AllCmds⍳⊂Cmd
      :Case 1 ⍝ 'File.Split'
          r←Split Input
      :Case 2 ⍝ 'File.Collect'
          r←Collect Input
      :EndSelect
    ∇

    Compress←{2⊃2(219⌶)83⎕dr ⍵}
    Decompress←{¯2(219⌶)⍵}
    Show←{~Arg.quiet:⍵}
    If←/⍨
    ∆FI←2⊃⎕vfi
    Bid←{2 ⎕NQ'.' 'getbuildid'⍵}

    ∇ r←Collect Arg;chunk;i;out;data;f;file;old;txt;path;t;name;len;crc0;totalLen;msg;TotalLens;CRCs;n;new;verify;Expand
    ⍝ Collect files in pattern file-999
     
    ⍝ Is this an info file?
      old←new←1⊃Arg.Arguments
      path←(-⊥⍨~'/\'∊⍨old)↓old        ⍝ where files are found
      Expand←Decompress⍣Arg.decompress
      :If 0≢f←Arg.newname
          n←'='=1↑f
          f←(n/path),n↓f
          new←f,((¯1↑f)∊'/\')/(⍴path)↓old
      :EndIf
      :If verify←'.info'≡¯5↑old
         ⍝ The info file should reside with all the files to collect
         ⍝ The info file may contain a different path which was recorded
         ⍝ when the original file was split.
         ⍝ We have to account for Classic:
          txt←⎕UCS 256|83 ¯1 ⎕MAP old
          'file is empty'⎕SIGNAL 11 If 0∊⍴txt
          txt←{1↓¨(⍵=1↑⍵)⊂⍵}¯1↓LF,txt~CR  ⍝ remove CRs, cut on LFs
          n←+/≠\'"'=f←1⊃txt               ⍝ find file name without path
          len←4+⍴name←('"'=1↑t)↓t←(-⊥⍨~'/\'∊⍨t)↑t←n↑f ⍝ include -NNN
          'invalid file format (filename not found)'⎕SIGNAL 11 If 0∊⍴name
          (totalLen crc0)←{n←¯1+' '⍳⍨r←(s←⍵⍳' ')↓⍵ ⋄ ((s-1)↑⍵)(n↑s↓⍵)}t←(n+2)↓f
          old←path,name ⋄ txt←1↓txt
          Expand←Decompress⍣(Arg.decompress∨1∊'compression used'⍷t)
          new←(¯5×'.info'≡¯5↑new)↓new,(∨/'/\'=¯1↑new)/name
          'invalid sequence of files'⎕SIGNAL 11 If(⍳⍴txt)≢∆FI⍕3↑¨('-'⍳⍨1⊃txt)↓¨txt
         ⍝ Find all file lengths and CRCs
          TotalLens←∆FI⍕(n←t⍳¨' ')↑¨t←(txt⍳¨' ')↓¨txt
          CRCs←n↓¨t
      :EndIf
     
      :If Arg.rewrite
          :Trap 22
              new ⎕NERASE new ⎕NTIE 0
          :EndTrap
      :EndIf
     
      Show'Collecting files to create ',new
      out←new ⎕NCREATE 0
     
      i←0
      :Repeat
          :Trap 22
              data←0⍴f←old,'-',,'ZI3'⎕FMT i←i+1
              Show'Processing ',f,'...'
              data←83 ¯1 ⎕MAP f
              :If verify
                  msg←'file ',f,' has wrong size'
                  :If (i⊃TotalLens)≠⍴data
                      msg←'file ',f,' has wrong CRC'
                  :OrIf (i⊃CRCs)≢Bid f
                      msg ⎕SIGNAL 11
                  :EndIf
              :EndIf
              (Expand data)⎕NAPPEND out,83
          :EndTrap
      :Until 0∊⍴data
     
      t←,⎕NSIZE out ⋄ ⎕NUNTIE out
      :If verify
          ('incorrect size for file ',new)⎕SIGNAL 11 If t≢∆FI totalLen
          ('incorrect CRC for file ',new)⎕SIGNAL 11 If crc0≢Bid new
      :EndIf
      Show'Done'⊣r←(⍕i-1),' files collected.'
    ∇

    ∇ r←Split Arg;tn;chunk;i;out;data;name;ok;f;approx;fsize;csize;suffix;n;ttn;tname;add;fmt;size;drop;t
    ⍝ Split a file into several smaller files
      :Trap 22
          fsize←⎕NSIZE tn←(f←1⊃Arg.Arguments)⎕NTIE 0
      :Else
          r←'** file "',f,'" not found' ⋄ →0
      :EndTrap
     
      csize←Arg.chunk
    ⍝ Is a specific number of files wanted?
      :If csize≡approx←0
          chunk←⌈fsize÷n←10 ⍝ no, default 10 files
      :Else                 ⍝ yes, what is it?
        ⍝ If suffix T/G/M is supplied it specifies an exponent of 10
        ⍝ AND it specifies the size of the individual files
          suffix←-1<chunk←10*12 12 9 9 6 6 3 3 0['tTgGmMkK'⍳¯1↑csize]
          approx←'~'∊1↑csize ⍝ do we want them roughly equal?
          ok←(0<+/n)∧(,1)≡1⊃(n n)←⎕VFI approx↓suffix↓csize
          'Invalid number of files'⎕SIGNAL ok↓11
          chunk←⌈fsize÷n←⌈chunk×n         ⍝ number and size of files
          :If suffix≠0 ⋄ n←⌈fsize÷chunk←n ⍝ recompute size of each file
          :AndIf approx ⋄ chunk←⌈fsize÷n ⋄ :EndIf
      :EndIf
     
      :If n=.>1 999
          ⎕NUNTIE tn
          'number of files must be from 2 to 999'⎕SIGNAL 11
      :EndIf
      Show r←'Split "',f,'" into ',(⍕n),' sections of ',(⍕chunk),' bytes'
     
      ttn←0 ⋄ add←{}
      :If 0≢tname←Arg.track
          :If tname≡1 ⋄ tname←f,'.info' ⋄ :EndIf
          ttn←tname ⎕NCREATE 0 ⋄ add←{(⍵,NL)⎕NAPPEND ttn}
          t←Arg.compress/' compression used'
          add'"',f,'"',(15⍕fsize),' ',(Bid f),t,' generated on ',⍕¯1↓⎕TS
      :EndIf
     
      fmt←,('<->,ZI',⍕3⌈1+⌊10⍟n)∘⎕FMT ⍝ we allow more than 999 for later, maybe
      drop←+/∨\⌽f∊'/\'
      :For i :In ⍳n
          data←Compress⍣Arg.compress⊢⎕NREAD tn,83,chunk
          out←(name←f,fmt i)⎕NCREATE 0
          data ⎕NAPPEND out,83
          size←⎕NSIZE out
          Show name,' (',((,'CI15'⎕FMT size)~' '),' bytes)'
          ⎕NUNTIE out
          add drop↓name,' ',(⍕size),' ',Bid name
      :EndFor
      ⎕NUNTIE tn,ttn
    ∇

    ∇ r←level Help Cmd;t
      r←⍬
      :Select AllCmds⍳⊂Cmd
      :Case 1 ⍝ 'File.Split'
          r,←⊂'Split a single file into (up to 999) smaller files'
          r,←⊂'    ]',Cmd,' <filename> [-chunk={nnn|<size>|~<size>] [-compress] [-quiet] [-track[=<logfile>]]'
          :If level>0
              r,←⊂''
              r,←⊂' -chunk=x            specifies the number of files of equal size to produce (nnn) or the maximum size of each file (size). Size consists of a value and a magnitude (K, M, G, T). Last file may be smaller unless size is prefixed by "~", in which case all files will have approximately same size, no larger than the specified size. The default for -chunk is "10".'
              r,←⊂' -compress           compresses the files using the zlib library'
              r,←⊂' -quiet              makes the command silent and no message appears while creating the files'
              r,←⊂' -track[=<logfile>]  writes log information to logfile (defaults to <filename>.info). If -track is given a file name the information will go to that file instead.'
     
              r,←'' 'Examples:'
              r,←⊂'    Produce 12 files named myfile-001 to myfile-012:'
              r,←⊂'        ]',Cmd,' myfile -chunk=12'
              r,←⊂'    Produce files of 10M (K, G, M, and T allowed) each up to the last one which may be smaller than 10M:'
              r,←⊂'        ]',Cmd,' myfile -chunk=10M'
              r,←⊂'    Produce files all about the same size under 10M (the last one may be smaller):'
              r,←⊂'        ]',Cmd,' myfile -chunk=~10M'
              r,←⊂'    Produce 10 files roughly the same size and a tracking file named myfile.info which can be used by ]Collect to recreate the file:'
              r,←⊂'        ]',Cmd,' myfile -track -quiet'
              r,←⊂'    Produce files of 1GB compressed and a tracking file named myfile.info which can be used by ]Collect to recreate the file. Note that you do not need a tracking file to use -compress but no verification will then be done:'
              r,←⊂'        ]',Cmd,' myfile -track -compress -chunk=1G'
              r,←⊂''
              r,←⊂'Once ]',Cmd,' has been used, ]Collect can be used to reassemble the original file.'
          :EndIf
     
      :Case 2 ⍝ 'File.Collect'
          r←⊂'Merge all the files with the specified path/name{-nnn} into a single file'
          r,←⊂'    ]',Cmd,' <filename> [-decompress] [-newname=<newname>] [-quiet] [-rewrite]'
          :If level>0
              r,←⊂''
              r,←⊂' -decompress         decompresses the files as they are processed. If a tracking file is used and compression was originally specified there is no need to specify this modifier as the tracking file will record this information'
              r,←⊂' -newname=<newname>  specifies the new name to use'
              r,←⊂' -quiet              makes the command silent and no message appears while creating the files'
              r,←⊂' -rewrite            erases the file if it exists'
              r,←'' 'Examples:'
              r,←⊂'    Collect \tmp\mypic.jpg-001, \tmp\mypic.jpg-002, ..., \tmp\mypic.jpg-nnn into \tmp\mypic.jpg:'
              r,←⊂'        ]',Cmd,' \tmp\mypic.jpg'
              r,←⊂'    Collect \tmp\mypic.jpg-001, \tmp\mypic.jpg-002, ..., \tmp\mypic.jpg-nnn into \tmp\great.jpg:'
              r,←⊂'        ]',Cmd,' \tmp\pic.jpg -newname=\temp\great.jpg'
              r,←⊂''
              r,←⊂'This user command is particularly useful when ]Split has been used on a file and the resultant files subsequently need to be reassembled.'
          :EndIf
      :EndSelect
      :If 0=level
          r,←''(']',Cmd,' -?? ⍝ for modifiers and examples')
      :EndIf
    ∇

    ∇ test track;ma;t
      ma←{p←⎕NEW ⎕SE.Parser'-chunk= -track[=] -rewrite -newname= -quiet' ⋄ p.Parse ⍵}
      {}÷'** file "dsa" not found'≡t←Split ma'dsa'
      Split ma'\tmp\from\ta.zip',track/' -t'
      Collect ma'\tmp\from\ta.zip -n=\tmp\to\'  ⍝,track/' -n='
      Collect ma'\tmp\from\ta.zip -n=\tmp\to\a.b'⍝,track/' -t'
      ⎕←'erase files now'
    ∇

:EndNamespace ⍝ split  $Revision: 1552 $
