﻿:namespace wsloc ⍝ V6.60
⍝ Locate strings in the workspace.
⍝ 2015 05 22 Adam: NS header and auto Version
⍝ 2016 05 08 DanB: modified APL name regex to account for :names at the beginning of a line
⍝ 2016 06 17 DanB: character left arg of SS always a name list and better doc
⍝ 2016 08 10 DanB: changed way regexes are found (PATMATCHrx) and added ability to "⍎" the argument
⍝ 2016 08 15 DanB: modified REPLX for patterns \\
⍝ 2016 08 29 DanB: bug fix
⍝ 2016 09 04 DanB: modified the way overlapping patterns are dealt with
⍝ 2017 05 08 Adam: Help overhaul
⍝ 2017 05 28 Adam: Desc casing
⍝ 2018 04 18 Adam: ]??cmd → ]cmd -??
⍝ 2018 05 08 Adam: help tweaks
⍝ 2019 02 04 Adam: help overhaul
⍝ 2020 06 30 Adam: avoid nonce errors

⍝ See 'Describe' for details.
⍝ When making modifications to this code you should run the test suite in \qa\ws\testlocate.dws

    (⎕IO ⎕ML ⎕PP)←0 2 16 ⋄ AllCmds←,⊂'Locate'

    ∇ r←Version;⎕ML
      ⎕ML←1
      r←⊢/∊'V'⎕VFI⊃⎕SRC ⎕THIS
    ∇

    ∇ r←List
      r←⎕NS¨0/¨AllCmds
      r.Name←AllCmds
      r.Group←⊂'WS'
      r[0].Desc←'Locate (and, optionally, replace) strings in the workspace'
      r[0].Parse←''
    ∇

    ∇ r←Run(Cmd Args)
      r←##.THIS SS Args
    ∇

    reFormat←{⍺←⎕pw ⋄ ⍺ ⎕se.Dyalog.Utils.(reshapeText∘condRavel)⍵}

      Help←{⍺←0
          x←Synopsis(≠⊆⊣)⎕UCS 13
          d←'Locate (and, optionally, replace) strings in the workspace',CR,0⊃x
          s←CR,']',⍵,' -??? ⍝ for complete documentation'
          i←CR,CR,']',⍵,' -??  ⍝ for syntax information and examples'
          ⍺=0:d,i,s
          ⍺=1:d,CR,(∊CR,¨5↓x),SwitchHelp,(∊CR,¨4↑1↓x),Examples,s
          Describe,i}

    (CR NL)←⎕av[3 2]

    Switches← '-objects= -class∊ ,~234 -recursive[=] -exclude∊BbTtCcLlGg -apl -insensitive -name -numeric -show -callback='
    Switches,←'-syntactic[∊]LlRr -pattern[=] -debug -nomessages -replace -return[=]count details each text -execute'

    _ ←'Searches (optionally specific types of items) for the strings specified in the argument. '
    _,←'By default the items searched are functions and operators. '
    _,←'The strings are separated by spaces in the argument. '
    _,←'Each line containing a searched string is displayed and a caret (∧) points to the exact location '
    _,←'where a match is found. If the string searched is a regular expression the entire match '
    _,←'is underlined with carets.'
    Synopsis←(CR,CR,_),⍨'    ]',(∊AllCmds),' <text> [<replacement>] [<text> [<replacement>]] ...','(-(\w{3})\w+\[?=)(]?)' '-\w+' ' Modifier ''(\w{3})\w+''' 'accepts values consisting' 'accepts only values'⎕R'[\1<\2>\3]' '[&]' '    <\1> ' 'must consist' 'must be one of'⊢17↓⎕se.SALTUtils.describeSwitches Switches

    _←CR,CR,'Modifiers:',CR
    _,←'    -callback=<cal>     call a user program each time a match is found',CR
    _,←'    -class=<cla>        specify the item name classes (for example, 2=variables) to look into (*)',CR
    _,←'    -debug              show debugging information',CR
    _,←'    -exclude=<exc>      exclude some elements from the search (*)',CR
    _,←'    -execute            execute the argument instead of taking it verbatim',CR
    _,←'    -insensitive        search case insensitive',CR
    _,←'    -name               include the name of the program next to each match shown',CR
    _,←'    -nomessages         do not display warnings',CR
    _,←'    -numeric            include numeric variables in search',CR
    _,←'    -objects=<obj>      specify which objects to search',CR
    _,←'    -pattern[=<pat>]    the strings to search are regular expression patterns',CR
    _,←'    -recursive[=<rec>]  search recursively from object specified',CR
    _,←'    -replace            replace mode',CR
    _,←'    -return[=<ret>]     return a result (*)',CR
    _,←'    -show               show hits',CR
    _,←'    -syntactic[=<syn>]  search syntactically (*)',CR,CR
    _,←'(*) Modifier value restrictions:'
    SwitchHelp←_
    _←CR,CR
    _,←'Examples:',CR
    _,←'    To search for the string "queens":',CR
    _,←'            ]Locate queens',CR
    _,←'            ∇ #.queens (3 found)',CR
    _,←'        [0]   queens←{⎕IO ⎕ML←0 1         ⍝ The N-queens problem.',CR
    _,←'              ∧                                   ∧',CR
    _,←'        [24] chars←''·⍟''[(↑⍵)∘.=⍳⍺] ⍝ char array of placed queens.',CR
    _,←'                                                          ∧',CR
    _,←'    To search for the string "queens" irrespective of case and ignoring comments:',CR
    _,←'            ]Locate queens -insensitive -exclude=C',CR
    _,←'            ∇ #.queens (1 found)',CR
    _,←'        [0]   queens←{⎕IO ⎕ML←0 1        ⍝ The N-queens problem.',CR
    _,←'              ∧',CR
    Examples←_


    ∇ r←GenHelp;syntactic
      r←↑,/CR,¨2↓2↓¨⎕NR'GenHelp'
⍝
⍝ ─── General ───
⍝
⍝ To search recursively for the string 'abc' in all functions in the active namespace:
⍝     ]Locate abc
⍝
⍝ ]Locate takes the following optional modifiers:
⍝
⍝ -exclude=      C T B L G
⍝     B: the object's Body (code)
⍝     C: the object's Comments
⍝     T: the object's Text within quotes
⍝     L: if localized (listed on line [0] of a traditional program or operator)
⍝     G: Global (NOT listed on line [0])
⍝
⍝ -apl           consider strings as APL code after ⍎, ⎕trap, etc.
⍝
⍝ -execute       execute the argument and use result as search phrase
⍝
⍝ -objects=      list of objects or components to look into, or {tie no[,start[,end component[,increment]]]}. Cannot be combined with -class.
⍝
⍝ -class=        list of classes (e.g. 3=functions) to look into. Only -class=9 implies -recursive. Cannot be combined with -objects.
⍝
⍝ -insensitive   insensitive (case) search
⍝
⍝ -numeric       include numeric (non character) variables in search. Search will still be performed as characters. Implied if phrase only includes numbers.
⍝
⍝ -pattern       the strings are single or multiline mode .NET patterns (see further below).
⍝
⍝ -return[=count|details|each|text]
⍝     ]Locate returns an empty result unless this modifier is used. Without it the program will
⍝ display the hits in the session log and the shape of the result is a matrix of 0 rows by
⍝ N columns where N is the number of hits, e.g. a 0 47 shaped matrix for 47 total hits.
⍝
⍝     -return=count returns a matrix of integers, one row per string, one column per object searched.
⍝ Each number represents the number of hits of a string for a given object.
⍝ To know to which string/object the number belongs to an extra row on top
⍝ shows the object and an extra column to the left shows the string searched.
⍝
⍝     -return=each returns a vector of enclosures,
⍝ each element being [1] the object name followed by [2] a 3D integer structure.
⍝ In that 3D structure each plane represents a hit.
⍝ The first row describes the hit with 5 numbers: 1. position in text, 2. length of hit, 3. searched string number, 4. row, 5. column
⍝ The other rows will only exist if a pattern with substrings was used,
⍝ each row belonging to a substring. The numbers have the same meaning.
⍝
⍝     -return=details is like -return=each with one more column for each row: the text found.
⍝ NOTE: the text of the line where it is found is not returned. It is easy enough
⍝ to get that information as part of the regular expression itself.
⍝
⍝     -return[=text] returns the entire output as a string. (May WSFULL.)
⍝
⍝ -recursive[=]  searches namespaces recursively from the namespace specified (default
⍝ current ns). Search is always recursive unless -class is set.
⍝
⍝ -nomessages    supress warnings and messages when -return is used.
⍝
⍝ -syntactic[=L R]
⍝ Finds only occurences NOT preceded (-syntactic=L) or followed (-syntactic=R) or surrrounded (-syntactic or -syntactic=LR) by
⍝ a letter, digit, ⎕, ∆, ⍙, _ or ¯ characters.
⍝
⍝ -show          shows hit-count when using -return or -replace.
⍝
⍝
⍝ ─── RESULT ───
⍝
⍝ The command will, by default, display the lines where the strings are
⍝ found with a caret (∧) below. It then returns an empty result. It
⍝ can also return an integer table with as many columns as objects
⍝ were searched and as many rows as strings were looked for with a row on
⍝ top with object full path names and one column to the left for the string
⍝ searched. For example, if all 20 functions of a workspace were searched
⍝ for 3 strings then a 4 by 21 table of the number of hits per program/string
⍝ with names would be returned. See section on -return=count above.
⍝
⍝
⍝ ─── EXAMPLES ───
⍝
⍝     ]Locate abc                             ⍝ search 'abc' everywhere
⍝     ]Locate abc def                         ⍝ search 'abc' and 'def' everywhere
⍝     ]Locate abc   -syntactic=R              ⍝ search names ending in '...abc'
⍝     ]Locate abc f -syn -exclude=TC -obj=x,y ⍝ syntactic search of 'abc' and 'f' in code (not  Text or Comments) of objects 'x' and 'y'
⍝
⍝ In the following example we have a file of ⎕CRs of functions and
⍝ we want to know where variables 'x' and 'yz' are used AND not
⍝ localized in components starting at 11 (file is tied with tie number 8):
⍝
⍝     ]Locate x yz -ex=LCT -apl -syn -obj=8,11  ⍝ look for 'x' and 'yz' as exact
⍝ names (-syn) in file tied to 8 starting at component 11. Don't look in
⍝ comments or text (-ex=CT) except where ⍎ is concerned (-apl) and
⍝ exclude where localized (-ex=L).
⍝
⍝     ]Locate x y z -ob=F,G -ins -syn -return ⍝ return a 4×3 of the number of hits in
⍝ each of <F> and <G> for lower and uppercase names 'x', 'y' and 'z'.
⍝ The table will include a extra column for the strings and extra row for the names.
⍝
⍝ To include spaces or dashes ( - the modifier delimiter) in the search, surround the text with
⍝ double or single quotes:
⍝
⍝     ]Locate '-/¨A B C'
⍝ will look for -/ of each of A, B and C. Otherwise the command will report an error
⍝ with the non existent "/¨A" switch.
⍝
⍝ To replace syntactic 'abc' by 'xyz' in all the code of each program in the current namespace:
⍝     ]Locate abc xyz -syn -replace -exclude=TC'
⍝
⍝
⍝ ─── PATTERN MATCHING ───
⍝
⍝ Pattern matching is possible to search and replace. Two ways are possible, either using
⍝ ⎕S or by using .NET (Windows only). ⎕S will be used by default. Select -pattern=DNET to use .Net
⍝ and/or -pattern=SINGLE to consider the entire program as a single string instead of separate
⍝ lines.
⍝
⍝ For example, we wish to find where files are tied (or created), that is, find
⍝ out which strings contain a filename. In regex terms: find strings that are "names separated
⍝ by / or \ possibly prefixed by //name or LETTER: and a /".
⍝ To do that we'd enter something like this:
⍝
⍝     ]Locate "'(//\w+/|[a-zA-Z]:/)?([/\\]?[.\w]++)+'"  -pattern -excl=BC
⍝   to look for a filename within text of all programs (EX=BC excludes body and comments).
⍝ Because we want to include the quotes in the search we surround the expression with quotes
⍝ and insert the whole thing inside double quotes for the parser.
⍝
⍝ To simplify search of names and numbers, the pseudo patterns ⍺ and ⍵ can be used.
⍝   For example to see where 3 names are assigned 3 numbers you could do:
⍝
⍝     ]Locate '⍺ ⍺ ⍺←⍵ ⍵ ⍵' -pattern
⍝
⍝ When displaying hits carets (∧) are shown under the whole match, not just the
⍝ beginning as in non-regex mode. Furthermore, if the match spans several lines, (can
⍝ only happen in -single mode) only the first line is displayed and the caret extends
⍝ 1 further to denote the fact. Ex: Given <F>:
⍝     ∇ F A
⍝  [1]  ⎕←A
⍝     ∇
⍝
⍝ Doing  "]Locate A.*A -p=single"  would find a match spanning from the 'A' in the header
⍝ to the 'A' on line 1 and the display would look like this:
⍝
⍝     ∇ #.F (1 found)
⍝  [0]   F A
⍝          ∧∧
⍝  The carets extend past 'A' to denote that the match continues further.
⍝
⍝ To use .Net use -pattern=dnet
⍝
⍝ If the pattern is too long to enter put it in a variable and use -execute, like this
⍝     exp←''((?(D)(?<noP>(?>(((&S)|[^()])*)))(?<I>((?>⍵)|(?>⍺)|(?&S)|(?&P))\s*(?&B)??)\s*)(?&I)'
⍝     ]Locate exp -pattern -execute
⍝
⍝
⍝ ─── CALL BACK user code ───
⍝
⍝ Whenever a match occurs it is possible to call a user function BEFORE the match is
⍝ displayed. To do so use the -callback switch, like this:
⍝
⍝     ]Locate a.*b -pattern -callback=MyPgm
⍝
⍝ If <MyPgm> is a valid program in the calling namespace it will be called every time
⍝ a match is found. The supplied right argument will be the string found. Its left
⍝ argument, if it accepts one, will be the name of the object, its location, its display
⍝ form and the matching information (position, length, pattern number, row and column).
⍝ If your program returns a single 1 the match will be displayed.
⍝
⍝
⍝ ─── REPLACEMENT ───
⍝
⍝ To perform replacement use pairs of search-string replacement-string and use
⍝ the -replace switch. If you use also the -show switch you will also see where
⍝ the changes are made.
⍝
⍝ For example to replace 'A' by 'BC' and 'D E F' by 'XYZ' everywhere you should do
⍝
⍝     ]Locate   A  BC    'D E F'  XYZ     -replace
⍝
⍝ The other modifiers apply as well, so using -exclude=C will exclude Comments when
⍝ performing replacement.
⍝
⍝ Pattern matching replacement is also possible. If specific sub-patterns
⍝ are to be reused you must use the backslash followed by their position number
⍝ to denote them. For example, to change all strings like 'Word Something Number'
⍝ into 'Number Something Word' you should do
⍝
⍝     ]Locate  "(\w+)(.*?)(\d+)"  "\3\2\1"  -replace -pattern
⍝
⍝ Here, in the string 'abcdef and 1234', group 1 (\w+) will match 'abcdef', group 2
⍝ (.*?) will match ' and ', group 3 (\d+) will match '1234' and the replacement expression
⍝ "\3\2\1"  will effectively switch them around, putting the 3rd string (\3) before the 2nd
⍝ one (\2) which will be put before the 1st one (\1).
⍝ Each substring must be surrounded by parentheses as usual to allow grouping (see the
⍝ regular expression documentation on ⎕R for details).
⍝
⍝
⍝ ─── LIMITATIONS ───
⍝
⍝ Will search functions, variables and files. This version also looks at namespaces
⍝ by default, unless the modifier -class= is used to limit the search to specific classes of objects.
⍝
⍝ Excluding locals or globals only makes sense for FULL syntactic search in programs.
⍝ The command will simply look for a match in the first line of a traditional program and
⍝ either continue or abort depending on the request.
    ∇

    ∇ r←Describe
      r←'>>>(.*)'⎕R{'──┤',3↓⎕PW↑⍵.Match,' ├',⎕PW⍴'─'}GenHelp
    ∇

    Cset←'_abcdefghijklmnopqrstuvwxyz∆',⎕A,'⍙',⎕Á,'ÀÄÅÆÉÑÖØÜßàáâäåæçèéêëíîïñóôöøùúûü0123456789⎕'
    Uset←3 27⍴Cset

    _←0≤⎕NC⍪⎕AV
    _←'a-zA-Z_',(_⌿⎕AV)~,⎕A,27⍴Cset
   ⍝ ?: is non capture, ?> is atomic grouping, ?<! is negative lookbehind
   ⍝ We could write it this way but .Net does not understand possessive quantifiers (+)
    AP←'(?>(?<!\s:)(?<!^:)(?<![⎕0-9',_,'])[',_,'][',_,'0-9]*+|⍺++|⍵++)'    ⍝ token name (⍺) pattern - no ⎕name
   ⍝ So we write it this way:
    AP←'(?>(?<!\s:)(?<!^:)(?<![⎕0-9',_,'])(?>[',_,'][',_,'0-9]*)|⍺⍺|⍺|⍵⍵|⍵)'    ⍝ token name (⍺) pattern - no ⎕name

    OP←'¯?(?<![',_,'0-9])',C←'(?>\d+\.?\d*|\d*\.?\b\d+)(?>[eE]¯?\d+)?'     ⍝ number (⍵) pattern
    OP←'(?>',OP,'(?>[jJ]¯?',C,')?)' ⍝ complex number

    Output←1 ⍝ produce output, set to 0 to limit output, for ex when testing

    ⎕EX¨'C_'

      remRef←{names←(b←0<cl←⍺.⎕NC n)/n←⍺.⎕NL ⍵         ⍝ ensure visible
          ~∨/n←9.1=b/cl:names                          ⍝ no namespaces
          (names/⍨~n),⍺{⍵/⍨(⍺≠s)∧⍺=(s←⍺⍎¨⍵).##}n/names ⍝ eliminate refs
      }

    namesOf←{⎕ML←3 ⋄ (~⍵∊', ')⊂⍵}

    Monadic←900⌶

    Msg←{0∊⍴⍵: ⋄ RETURN≢1:⎕←⍵ ⋄ ⍙Rz,←⍵,CR} ⍝ can be eventually changed to account for ]ucmd call

    ∇ ⍙Rz←{LA}SS RA;t1;btc;Rstr;var;id;NAME;NOMESSAGES;SHOW;DEBUG;PATTERN;INSENSITIVE;RETURN;APL;RECURSIVE;REPLACE;EXCLUDE;SYNTACTIC;NUMERIC;CLASS;⍙98;⍙97;onfile;⍙95;⍙94;⍙93;⍙92;⍙91;⍙90;Strings;inc;CurSpace;t;Path;OBJECTS;Pattern;pcr;name;action;lac;SALTed;Typ;out;DF;m;first;loc;∆;b;x;L;A;Dots;LAisNs;EXECUTE;CALLBACK
      ⍙93←'Search String (Find and Replace) V',2⍕Version
     ⍝ This program will perform String Search and Replace on most APL objects.
     ⍝ It receives a ns as Rarg which contains the Strings to search/replace and the switches OR a string which it parses itself.
     ⍝ The left arg is either a ns to search into OR a string of the objects to search in the current ns
     
     ⍝ Syntax:
     ⍝        [namelist] SS 'strings [-switches]'  ('-' is the delimiter to use for switches)
     
     ⍝ NOTE: this version does not replace numbers in numeric vars, nor searches instances.
     
     ⍝ It first defines a series of functions to do the work properly, in this ns, then proceeds to do the work.
     ⍝ If it has to do recursive searches it reuses the functions already defined. Many objects temporarily defined start with a '⍙'.
     
     ⍝ The Larg is the list of objects to search (default all). If it is a ref we search the objects in it.
      DF←{(⍵.⎕DF df){⍵}⍕⍵{⍺}df←⍵.⎕DF ⎕NULL}
      onfile←0 ⋄ id←DF CurSpace←''⍴⎕RSI ⍝ do NOT use '⎕se' ⎕WG 'curspace' as this can be called from anywhere
      :If LAisNs←9∊lac←⎕NC'LA' ⋄ id←DF CurSpace←LA        ⍝ a ref can be presented as such
      :ElseIf 2∊lac                                       ⍝ is LA defined?
         ⍝ If defined, the left arg could be integers for tie #, components to search
          :If ~onfile←∨/1 3∊t←10|⎕DR LA
            ⍝ Otherwise it could be Space, ID  or a list of names
              :If LAisNs←((,2)≡m←⍴LA)>b←t∊0 2 ⋄ (CurSpace id)←LA   ⍝ space, ID (2 elems, not character)
              :EndIf
          :EndIf
      :Else ⍝ LA is undefined, we default to the current ns defined above
          LAisNs←1
      :EndIf
      Path←id,'.'
     
     ⍝ Right arg may be a single 1 in which case the left arg is a ns to search
      :If 1≡RA                      ⍝ a single 1 means 'recursive call'
          ⍙Rz←⍙R0                   ⍝ ⍙R0 and ⍙ are global to this program and hold common data needed at each (recursize) call
          REDEFINE ⍙                ⍝ redefine all vars locally
          LA←CurSpace remRef-CLASS  ⍝ the objects to  search
     
      :Else                         ⍝ proceed to the definition of the work to do
     ⍝ Left arg is a Namelist or tie # and right arg is either a properly formed string with switches...
          :If var←9≠⎕NC'RA'         ⍝ 'var' needed in DEBUG
             ⍝ We require the Parser to be there
              ⍙←(⎕NEW ⎕SE.Parser(Switches'upper')).Parse RA ⍝ args and switches kept in this global
          :Else ⋄ ⍙←RA
          :EndIf
     
         ⍝ RECURSIVE may specify a starting point where to recurse from
          :If 1=⍴⍴t←⍙.RECURSIVE ⋄ CurSpace⍎←t ⋄ Path←(id←⍕CurSpace),'.' ⋄ :EndIf
          ⍙.RECURSIVE←(t≡1)∨0≡⍙.CLASS ⍝ 2013/7 always recursive now unless -class is set
          ⍙.(EXCLUDE←∨⌿2 5⍴'BTCLGbtclg'∊EXCLUDE)
          ⍙.(SYNTACTIC←∨⌿3 2⍴'lrLR11'∊⍕SYNTACTIC)
     
         ⍝ If a callback program is wanted we prepare it here
          :If RZ←0≢t←⍙.CALLBACK     ⍝ execute function for each match found
              RZ←×t1←2 ⋄ t←' '~⍨((~'#'∊t)/⍕CurSpace,'.'),t ⍝ prepend calling ns if relative
              :If ~(⎕NC⊂t)∊3.2 3.3 ⋄ (RZ t1)←|2↑0⊃⎕AT t ⋄ :EndIf
              'Function to execute must be monadic or dyadic'⎕SIGNAL t1↓11
              CB←⍎((t1=1)/'⊢∘'),t   ⍝ keep global for recursive calls
          :EndIf
     
          :If ⍙.Pattern←Pattern←0≢t←⍙.PATTERN
          :OrIf t←⍙.APL
         ⍝ Pattern may be specified as SINGLE (otherwise multi defaulted) or DNET
              :If t≡1 ⋄ ⍙.PATTERN←3 ⍝ w/o a value PATTERN means the default (⎕S)
              :Else ⍝ engine and/or mode were specified
                  :Trap 701
                      ⍙90←(⎕NEW ⎕SE.Parser(' dnet single' 'upper allownospace')).Parse b\t/⍨b←','≠t
                  :Else
                      'Unknown value for -pattern=; use DNET and/or SINGLE'⎕SIGNAL 11
                  :EndTrap
                 ⍝ 2=use ⎕S, 4=use .Net + 1 if multiline (the default)
                  ⍙.PATTERN←3 5[⍙90.DNET]-⍙90.SINGLE ⍝ 0 2⊤→fn,multi
              :EndIf
          :EndIf
          ('Only one of ',('name list, '/⍨~LAisNs),'-Objects or -Class is acceptable')⎕SIGNAL 11 if 1<+/(lac=2),0≢¨⍙.(OBJECTS CLASS)
         ⍝ Here all switches have been defined in global '⍙'
          REDEFINE ⍙
          :If DEBUG
             ⍝ Display debugging information
              ⍙90←((5 5 6/⍙90)/'FULL Left Right '),(∨/⍙90←<\(∧/⍙90),⍙90←SYNTACTIC)/'Syntactic '
              ⍙92←(((⍙92⍳'-')-var×⍴⍙92)↑⍙92←'1 letter options-switches'),' method.'
              ⍙90←'program called in ',⍙90,((6-REPLACE×⍴⍙92)↑⍙92←'SEARCHREPLACE'),' mode using the ',⍙92,CR
              ⍙90,←'Called from ',id,CR
              ⍙90,←'Number of Strings: ',CR,⍨⍕⍴Strings
              ⍙90,←(∨/EXCLUDE)/('Exclude',(5 5 9 10 8/EXCLUDE)/' BODY TEXT COMMENTS LOCALIZED GLOBALS'),CR
              t←('count' 'details' 'each'⍳⊂RETURN)⊃'a count of' 'individual detailed' 'individual' 'text'
              ⍙90,←(0≢RETURN)/'The program returns ',t,' results',CR
              ⍙90,←APL/'APL mode ON',CR
              ⍙90,←INSENSITIVE/'Case insensitive mode ON',CR
              ⍙90,←NUMERIC/'Numeric vars searched',CR
              ⍙90,←RECURSIVE/'Recursive mode ON',CR
              t←(1⍴⍙92←,0 2⊤PATTERN)⊃0 '⎕S' '.Net'
              ⍙90,←(0<PATTERN)/t,' patterns are used ',((~1↓⍙92)/', singleline mode '),CR
              ⍙90,←(~NOMESSAGES)/'Messages are shown',CR
              ⍙90,←SHOW/'Hits are displayed',CR
              ⍙90,←EXECUTE/'Argument is an expression to execute (⍎)',CR
              ⍙90,←(CALLBACK≢0)/'Program ',CALLBACK,' is called for each match',CR
              Msg ⍙90
          :EndIf
          :If Rstr←REPLACE   ⍝ is this a replacement?
              'ERROR: Argument length is odd'⎕SIGNAL(2|t←⍴Strings)⍴11 ⍝ Do we have only pairs then?
              ⍙.Rstr←Rstr←(t⍴0 1)/Strings
          :EndIf
     
          ⍙Rz←⍬ ⍝ initialize here in case Msg uses it
         ⍝ Issue a warning if EXCLUDE locals is used with non full SYNTATIC search:
          Msg((≠/¯2↑EXCLUDE)∧(≠/SYNTACTIC)∧DEBUG≥NOMESSAGES)/'** Partial SYNTACTIC search used with local exclusion: results can be misleading'
     
         ⍝ We set the Numeric flag if all numeric
          ⍙.NUMERIC←NUMERIC←NUMERIC∨∧/0⊃⎕VFI⍕Strings
     
         ⍝ Use proper pattern matching fn
          var←'⍷' ⍝ the fn to use to search
          :If APL∨×t←PATTERN ⋄ var←(⍕2|t),'∘PATMATCH',t⊃2/0 'rx' 'dnet' ⋄ :EndIf
          ⍙Find←⍎var
     
          x←'~'∊1↑CLASS ⍝ eXclude?
          CLASS←(t∨0 1 1>∨/t←∨⌿3 3⍴'234VFOvfo'∊CLASS)/t1←2 3 4 ⍝ default to programs/operators
          ⍙.CLASS←CLASS←(t1∘~⍣x⊢CLASS),RECURSIVE/9
          :If onfile<LAisNs  ⍝ if not on file AND the LA is a space
            ⍝ -objects may specify tie #, cpts to check; they can be separated by commas
              (onfile LA)←{⍵≡0:0 ⋄ ∧/⎕IO⊃v←' ,'⎕VFI ⍵:1,1↓v ⋄ 0 ⍵}OBJECTS
              :If (0≡LA)∨t←'~'∊1↑LA
                 ⍝ Do we need ns as well? the recursive method will include them w/o refs
                  LA←(CurSpace remRef-CLASS)~t/namesOf⍕1↓LA ⍝ no patterns (yet)
              :EndIf
          :EndIf
     
         ⍝ LA could be tie #, start, last cpts, increment
          :If onfile
              ⍙92←⍙91⌈t←LA,(⍴,LA)↓⍙91←0,(2↑⎕FSIZE 1↑LA),1
              ⍙92[2]←⍙91[2]⌊t[2]+1 ⋄ LA←⍙92[⍳3]-inc←0,⍙92[3],0
          :Else
            ⍝ Ensure left arg is list of objects (VTV)
              :If 1=≡,LA ⍝ allow enclosed list
                  LA←namesOf,',',LA ⍝ can be a matrix or space delimited string
              :EndIf
          :EndIf
     
         ⍝ The rule for SHOW and RETURN is like this:
         ⍝ If we are in search mode (not replace) then by default we show and return nothing.
         ⍝ If return is not to show (show is 1, 0 or 'text') we DON'T show (if SHOW is set we leave it ON)
         ⍝ If we are in replace mode the conditions are reversed.
         ⍝ In short, we show the hits if requested or a result is wanted and no replacement is made
          ⍙.SHOW←SHOW←SHOW∨REPLACE<0 1 'text'∊⍨⊂RETURN
          :If REPLACE ⋄ RETURN←0 ⋄ :EndIf ⍝ no other RETURN makes sense in replace mode
     
     
     ⍝ ----------- Define Local functions. Fixed utilities start with a lowercase letter. -------------
     
     ⍝ The shape of the result depends on the request:
     ⍝ by default, the program returns a (0,totalHits)⍴0. It may return a list of enclosed series
     ⍝ of a program name followed by a table of integers (3D): ith match, pos, length, row, col.
     ⍝ It can also return the number of hits for each string in each objects (2D int) with titles.
     ⍝ ⍙Rz is initialized accordingly: (exit if nothing to search)
     
          ⍙MrG←0 1∘↓⍣(RETURN≡'count') ⍝ remove strings column of recursive results
          ⎕FX(⊂'r←l GETSTRINGS r'),('details'≡RETURN)/⊂'r←r,(⊂¨r[;;0]+⍳¨r[;;1])⌷¨⊂l'
     
          :Select RETURN
          :CaseList 'details' 'each'
              ⍙Rz←⍬                   ⍝ each enclosed result catenated, result is vector
              ⍙93←⎕FX'r←r ⍙cAr new' 'r←r,⊂2↑new'
          :Case 'count'
              t←(⍴Strings)÷1+REPLACE  ⍝ number of Strings
              ⍙Rz←(1+t,0)⍴' ',Strings{⍺/⍨(⍴⍺)⍴1,~⍵}REPLACE    ⍝ each hit # returned for each searched string
              ⍙93←⎕FX'r←r ⍙cAr (f new x)'('r←r,(⊂f),+⌿new[;0;2]∘.=⍳',⍕t)
          :Case 0
              ⍙Rz←0 0⍴0               ⍝ nothing returned (0×ttl hits returned)
              ⍙93←⎕FX'r←r ⍙cAr (f new x)' 'r←r,(0,1↑⍴new)⍴0' ⍝ final shape will 0xTTL hits
          :Else ⍝ 1 or 'text'
              RETURN←1  ⍝ ⍙Rz already set above
              ⍙93←⎕FX'r←r ⍙cAr (x y new)' 'r←r,new'
          :EndSelect
          →(⍴Strings)↓⍙enD           ⍝ anything to search?
     
          :If INSENSITIVE ⋄ ⍙.Arguments←Strings←LowerCase¨Strings ⋄ :EndIf ⍝ insensitive to case?
     
          Msg(NOMESSAGES<0∊∊⍴¨Strings)/'* Empty Strings are ignored'
     
          ⍙R0←⍙Rz                    ⍝ initial value (used when recursing)
     
          ⍙.EXECUTE←0                ⍝ turn that OFF for recursive calls
      :EndIf
     
     ⍝ The "actions" table: this version does not replace numeric values
     ⍝ Each row tells what can be done: add to result, allow search, allow replace
      btc←'CSENI',1,5 2⍴1,REPLACE,1,REPLACE,1,REPLACE,NUMERIC,0 0 0 ⍝ table of actions for each case
     ⍝
     ⍝ ===========================================================================================================
     ⍝
     ⍝                                   --------     Main loop     --------
     ⍝
      first←1                                ⍝ is this the first item shown?
      :While ~0∊⍴LA
          id←Path,name←⎕IO⊃LA                ⍝ next name to search
          ⍙93←⎕EX'⍙92'                       ⍝ better be safe than sorry
          pcr←0                              ⍝ proper ⎕CR?
          :If var←onfile
              →(≥/1↓LA←LA+inc)⍴⍙enD          ⍝ over with the file?
              id←'Cpt ',(⍕1↓2↑LA),': '       ⍝ remember path for display
              ⍙92←⎕FREAD 2↑LA                ⍝ fetch the object to search from file
          :ElseIf var←∨/9 2∊CurSpace.⎕NC name
              ⍙92←CurSpace⍎name              ⍝ should not be shared!
          :Else                              ⍝ it is a function (or else, even undefined!)
              pcr←1=≡⍙92←CurSpace.⎕CR name   ⍝ we may have a special format for programs
          :EndIf
     
     ⍝ We have the object.
     ⍝ If it's character or numeric (and wanted): we can search AND replace.
     ⍝ If it's enclosed we only show if it's complex. If it's a ns we recurse if desired.
          :If 'P'=Typ←CurSpace name TYPE ⍙92
         ⍝ Types are: Scripted, Char, Num, Packed (NS), Enclosed and Invalid (like Null)
              t←⍙R0
              :If ⎕THIS≢CurSpace ⍝ avoid infinite loops
                  t←⍙MrG ⍙92 id SS 1
              :EndIf
              ⍙Rz←⍙Rz,t
              action←0 0 0 0
          :Else  ⍝ to replace we must either have a char matrix or a list of char vectors; none must contain CR/NLs
              :If SALTed←Typ='S'
                  :If SALTed←9∊⍙92.⎕NC'SALT_Data' ⋄ SALTed←{⍵.({⍵(⍎¨⍵)}⎕NL-2)}⍙92.SALT_Data ⋄ :EndIf
                  ⍙92←62 ⎕ATX'⍙92'
              :EndIf ⍝ change classes' code into a string
              action←btc[btc[;0]⍳Typ;]
              t1←∨/((⍴⍴⍙92)↓0 2 1)=⍙90←|≡⍙92
              :If t1∧2=⍙90 ⍝ enclosed, could be anything. The only way to tell:
                  :If 1∨.<∊⍴¨⍴¨⍙92           ⍝ if  all vectors or scalars
                  :OrIf ~∧/(10|,⎕DR¨⍙92)∊0 2 ⍝ and all character, then all is well
                      t1←0                   ⍝ otherwise can't replace
                  :EndIf
              :EndIf
              ⍝ There are more constraints so we can replace
              :If t1∧2∊⍙90,⍴⍴⍙92         ⍝ must be deep 2 OR matrix
              :AndIf t1←~∨/(NL,CR)∊⍕⍙92  ⍝ no such chars allowed
              :AndIf 2∊⍙90               ⍝ enclosures only
                  ⍙92←1↓∊CR,¨⍙92
              :EndIf
        ⍝ One more thing: if in REPLACE mode, a program can only be dealt with if it's not a ref
              :If REPLACE∧pcr ⋄ t1←name≡(⎕NS'').⎕FX ⍙92 ⋄ :EndIf ⍝ is it the same name?
              action[3]∧←t1 ⍝ still replace?
          :EndIf
     
         ⍝ We determine what to do: skip this item, replace, add to result
          ⍙93←0 1 5⍴⍴out←'' ⍝ default value
          :If action[2]∧REPLACE≤action[3] ⍝ don't search if REPLACE and not allowed to do so
             ⍝ Search & count: <SEARCH> returns an INT table of pos, len, hit#, line #, char #
              :If 0<⍙98←''⍴⍴⍙93←⍙91 GETSTRINGS(Typ,var↓'F')SEARCH ⍙91←1↓(,⌽∨\⌽' '≠⍙91)/,⍙91←CR,⍕⍙92
                  :If 0≢CALLBACK ⍝ call program for each match found
                  ⍝ If a result is expected we run the program on each match
                      :For m :In ↓⍙93[;0;]⊣t←⍬
                          t,←{RZ↓6::0 ⋄ (,1)≡1/(id CurSpace ⍙91 ⍵)CB ⍙91[⍵[0]+⍳⍵[1]]}m
                      :EndFor
                      ⍙93←t⌿⍙93
                  :EndIf
                  ⍙90←~0∊⍴⍙93
                  ⍙94←⍴⍙92
                  :If action[3]∧⍙90 ⍝ replace
                      (⍙93 ⍙91)←⍙93 REPL ⍙91
                      ⍙92←⍙94 BOX ⍙91
                      :If onfile
                          (⍙94 BOX ⍙91)⎕FREPLACE 2↑LA
                      :ElseIf var
                          t1←⍙94 BOX ⍙91
                          :If ∨/⍙90←'SE'=action[0] ⋄ t1←1↓¨(1,t1∊CR)⊂';',t1 ⋄ :EndIf
                          :If ⍙90[0] ⋄ t1←0 CurSpace.⎕FIX t1 ⋄ :EndIf
                          t←{CurSpace⍎name,'←⍵'}t1
                          :If 0≢SALTed             ⍝ restore SALT tag
                              'SALT_Data't.⎕NS''
                              t.SALT_Data.{⍎(⍕⎕IO⊃⍵),'←⎕io⊃⌽⍵'}SALTed
                          :EndIf
                      :Else   ⍝ it is a function. If it fails we need to adjust the search results:
                          ⍙93←(3≠10|⎕DR CurSpace.⎕FX ⍙92)⌿⍙93
                      :EndIf
                  :EndIf
                  ⍙90←~0∊⍙98←≢⍙93
                  ⍙90∧←REPLACE≤action[3] ⋄ ⍙93←⍙90⌿⍙93 ⍝ Effective replacement?
     
                  :If SHOW∧⍙90
                     ⍝ Display header
                      ⍙90←3 6⍴'    ∇ ',((6×¯1*Typ='S')↑'  VAR Space '),'Rep ∇ '
                      ⍙94←'∇'∊7⍴⍙91 ⋄ ⍙90←⍙90[var×2*⍙94;],id
                      out←first↓CR,CR,⍙90,1⌽') (',(⍕⍙98),' found'
                     ⍝ We find the location of each line and their lengths
                      t←0,t/⍳⍴t←CR=⍙91 ⋄ loc←(1+t),⍪¯2-/t,⍴⍙91
                      ⍙90←∪⍙93[;0;3]
                      :While 0<⍴⍙90
                         ⍝ Display lines found here
                          ⍙95←(~⍙94)/(NAME/id),(5⌈⍴⍙95)↑⍙95←'[',(⍕⍙97←0⊃⍙90),'] '
                          m←1+⍴L←⍙95,RTB,⍙91[↑+∘⍳/loc[⍙97;]-0 1=×⍙97] ⍝ the Line
                         ⍝ For patterns we display the whole match
                          t←⍙93[;0;3]=⍙97 ⋄ ⍙92←(t/⍙93[;0;4])PIO(t/⍙93[;0;1])*Pattern
                          A←m⍴' ' ⋄ A[(⍳m)∩⍙92+⍴⍙95]←'∧' ⍝ the ^ line
                         ⍝ If the display exceeds the printing width we try to trim edges
                          :If SHOW∧⎕PW<m-' '∊¯1↑A                     ⍝ edge condition
                              Dots←' ... '                            ⍝ change this need be
                              ∆←-5⌈⌊0.5×((⎕PW-t+⍴⍙95)÷+/A='∧')-t←1+⍴Dots
                              t←(m↑(1+⍴⍙95)⍴1)∨∆⌽↑∨/(⍳1+¯2×∆)⌽¨⊂A='∧' ⍝ where want to see the code
                              ∆←¯2-/∆/⍳⍴∆←1,⍨m←1,2≠/t                 ⍝ the length of each section
                          :AndIf ∨/b←6<∆×~x←(⍴∆)⍴1 0                  ⍝ any fits the bill?
                              t←m\x←x∧¯1↓1,x←x∨b ⋄ b←x/b              ⍝ where the new sections start
                              ⍙98←t⊂L,' ' ⋄ (b/⍙98)←⊂Dots ⋄ L←∊⍙98    ⍝ trim the line
                              m←t⊂A ⋄ (b/m)←1↑0⍴⊂Dots ⋄ A←∊m          ⍝ and the ^ line
                          :EndIf
                          out,←CR,L,CR,RTB A
                          ⍙90←1↓⍙90
                      :EndWhile
                      :If Output∧RETURN≢1 ⋄ ⎕←out ⋄ :EndIf ⍝ gather result or display now
                      first←0
                  :EndIf
              :EndIf
          :EndIf
          :If action[1]
              ⍙Rz←⍙Rz ⍙cAr id ⍙93 out
          :EndIf
          LA←(~onfile)↓LA
      :EndWhile
     
     ⍙enD: :If (RA≢1)∧REPLACE
          ⍙Rz←(⍕1↓⍴⍙Rz),' replacements made.'
      :EndIf
      ⎕EX lac↓'⍙'
    ∇

    isChar←{∨/0 2∊10|⎕DR 1/⍵}

    ∇ m←s BOX v;b;d;l;⎕IO
     ⍝    <BOX> fn: this function serves many purposes. It basically ⎕BOX the right argument.
     ⍝     If the right arg is NOT already a matrix then:
     ⍝     If called with a 0   left arg it exits AFTER boxing, using the 1st element as delimiter.
     ⍝     If called with a 1=⍴ left arg it exits BEFORE boxing, returning its right arg.
     ⍝     If called with a 2=⍴ left arg it boxes the right arg using CR then tries to match the original shape.
      ⎕IO←1 ⋄ →(2∊⍴m←v)⍴0 ⋄ →(1∊⍴s)⍴0 ⋄ m←⍬⍴v ⋄ →(s≡⍬)⍴0 ⋄ d←1↑v←((2∊⍴s)⍴CR),v
      l←1↓l-1+¯1⌽l←(~b)/⍳⍴b←d≠m←v,d ⋄ m←(⍴b)⍴(,b←l∘.≥⍳⌈/0,b←l)\b/m
      →(s≡0)⍴0 ⋄ v←s⌈⍴m←(0,-l⊥l←' '∧.=m)↓m ⋄ m←v↑m
    ∇

    ∇ Clean
      ⎕EX'⍙'⎕NL⍳10
    ∇

    ∇ R←bo HIDE M;L;C;T;B;A;t;p;apl;x;msk;q
⍝ Find search mask. M is text string. bo is keep body mask.
⍝ R=1 means valid search position.
      L←1,1↓msk←M∊NL CR          ⍝ find CRs and line starts
      R←(⍴msk)⍴1 ⍝⍲2|PATTERN            ⍝ final mask excludes NL if multiline
      :If APL∧'∇'∊7↑M            ⍝ is this a program?
          R←R∧¯1⌽L PartOS M∊'∇]' ⍝ then account for decorators
      :EndIf
⍝ Find Body, Comments and Text masks
      →0↓⍨∨/x←3↑EXCLUDE          ⍝ any code exclusions? (Body, Text, Comments)
⍝ Separate Body, Comments and Text masks
      B←C⍱T←T>C←L PartOS(M∊'⍝')>T←T∨L PartUS T←q←M=''''
      :If APL∧1∊T ⍝ do we have to take executable strings into account?
    ⍝ This is the pattern to look for for executable strings:
          apl←'(?i:⎕(EX|NC|CR|VR|NR|OR|AT|SIZE|NS|CS|ED|PATH|LOCK|TRAP←[^''⍝⋄\n\r]*''[ESNC]'')|⍎[∧''⍝\n\r⋄]*) *''((''''|[^''\r\n])*''[^⍝⋄\n\r]*)'
    ⍝ We extract any text to further search into
          t←⍙Find apl M 0 R(0 0)
          t←B[t[;0;0]]⌿t ⍝ remove those not in Body
      :AndIf 0<⍴p←t[;2;0]PIO t[;2;1] ⋄ msk[p]←1
    ⍝ Remove double quotes; careful not to remove too many
          msk←msk>q>q PartUS⍨q>¯1⌽q
          A←msk\1 HIDE msk/M
          B←C⍱T←T>A
      :EndIf
      :If bo ⋄ R←R∧B
      :Else ⋄ R←R>↑∨/x/B T C
      :EndIf
    ∇

    ∇ r←LowerCase r;b
     ⍝ Also works for Underbarred letters
      →(∨/b←r∊1 0↓Uset)↓0 ⋄ b←b/⍳⍴b ⋄ r[b]←Uset[0;27|(,1 0↓Uset)⍳r[b]]
    ∇

    ∇ z←a PartAS b
    ⍝ Partitioned ∧\
      z←~≠\a\z≠¯1↓1,z←(a←a≥b)/b
    ∇

    ∇ R←S PartOS P;T
     ⍝ Partitioned ∨\
      S←S∨P ⋄ R←≠\S\T≠¯1↓0,T←S/P
    ∇

    ∇ r←s PartUS p;t
     ⍝ Partitionned ≠\ (unequal scan)
      r←≠\p≠s\t≠¯1↓0,t←s/≠\¯1↓0,p
    ∇

    ∇ (ng pattern)←ShortCuts pattern;sl;e;good;caret;op;ap
     ⍝ Replace ⍺ and ⍵ by APL patterns
      caret←'[',⎕UCS 94   ⍝ map ∧ to ASCII caret if no translation done
     ⍝ For APL code we accept ⍺ & ⍵ as shortcuts for 'object name' and 'number'
      (ap op)←{(1+'\'=⍵)/⍵}¨AP OP
      pattern←'\\\\' '\\⍺' '\\⍵' '\[\∧' '⍺' '⍵'⎕R'\\\\' '⍺' '⍵'caret ap op⊢pattern
      ng←1+≢'\(\?<(?![=!])|(?<!\(\?)(?<!\\)\((?!\?)'⎕S 0⊢pattern  ⍝ how many parentheses?
    ∇

     ⍝ The next 2 functions are used to find where the pattern is found.
     ⍝ They return an int list indicating the start & length of each match.
     ⍝ Each subexpression specified is shown for each match (3D result).
     ⍝ They are ⎕io independent.
     ⍝ The pattern may not overlap.
     ⍝ ls/rs are left/right "syntactic" op. They work with 'notalpha' (see below).
     ⍝ 'mask' is where matches must occur entirely.

    ∇ r←{options}PATMATCHrx arg;pattern;string;notalpha;syn;mask;⎕IO;⎕ML;⎕WX;hits;z;good;cr;linemode;np
     ⍝ Pattern matching using ⎕S. Option is 0=single document, 1=multi-line
     ⍝ If single document we work in Document mode where EOL is recognized.
     ⍝ If not, we should work in Line mode ⍝and keep track of the length of each line.
     ⍝ If a mask is supplied we should work in Overlap Mode in order to eliminate matches
     ⍝ that occur in the masked out portion.
      arg←arg,(⍴arg)↓0 0 0 '' 0
      (pattern string notalpha mask syn)←,¨arg
      options←(options=0 0 1)/('DotAll' 1)('Mode' 'D')('Mode' 'M')
      :If 0∊mask←mask,((0∊⍴mask)×⍴string)⍴1 ⍝ any restriction?
         ⍝ we would think that "pattern←'(?!\001)',pattern" could be used to skip over masked positions
         ⍝ but because of ...\K we could miss cases, so we don't do that
          string[(~mask)/⍳⍴mask]←⎕UCS 1 ⍝ try to make ⎕S miss those positions
          options,←⊂'OM' 1              ⍝ Overlap Mode
      :EndIf
      (np pattern)←ShortCuts pattern
      r←(0,np,2)⍴0 ⍝ default result
      →0↓⍨≢hits←pattern ⎕S{⍵.(Offsets,[0.1]Lengths)}⎕OPT options+string
     ⍝ Screen out masked positions
     ⍝ If there is no possibility of overlap we do them all at once:
      np←+/z←⊃2⍴¨hits
      :If ∧/(¯1↓np)<1↓z[;0] ⋄ good←1
          :If ∨/syn
              good←notalpha[z[;,0],1+np]∧.≥syn
          :EndIf
          good>←0∊¨z[;1]↑¨z[;0]↓¨⊂mask
          r←good/hits
      :Else
          r←⊂⍤2⊢r
          :Repeat
              z←,0⊃hits ⋄ good←1
         ⍝ notalpha is where no name forming chars are, this option should normally not be set with a regex
              :If ∨/syn ⍝ syntactic search?
                  good←notalpha[0 1++\2⍴z]∧.≥syn
              :EndIf
              good←good>0∊mask[z[0]+⍳z[1]]
              r←r,good⍴hits ⋄ hits←1↓hits
              mask[z[0]+⍳z[1]×good]←0 ⍝ do not allow overlap
          :Until 0∊⍴hits
      :EndIf
      r←⊃r
    ∇

    ∇ r←{options}PATMATCHdnet arg;⎕USING;CR;LF;syn;mo;np;len;ind;pattern;string;notalpha;mask;ls;rs;⎕IO;good;code;e;sl;ptr;regcomp;regerror;regexec;regfree;sxi;z;C;NULL;⎕ML;⎕WX
      arg←arg,(⍴arg)↓0 0 0 '' 0
      (pattern string notalpha mask syn)←,¨arg
      mask←mask,((0∊⍴mask)×⍴string)⍴1
     
      ⎕WX←3 ⋄ ⎕ML←⎕IO←0
      (CR LF)←⎕AV[3 2]
      (r pattern)←ShortCuts pattern
      r←0⍴⊂(r,2)⍴0 ⍝ default result
     
     ⍝ Determine the options (0=singleline, 1=multiline)
      ⎕USING←'System.Text.RegularExpressions,system.dll'
      options←options⊃RegexOptions.(Singleline Multiline)
     
⍝ For multiline to work under .Net we need NLs
      ((string=CR)/string)←LF
     
      mo←Regex.Match string pattern options
     
     ⍝ Execute the expression until nothing found
      :While good←mo.Success
          (len ind)←mo.(Length Index)
          :If ∨/syn
     ⍝ notalpha is where no name forming chars are (padded 1 left and right to avoid INDEX error)
     ⍝ This supplements the ability of the engine to detect word characters with \w as Dyalog
     ⍝ also uses other characters like Á. 'notalpha' can indicate where those are.
              good←notalpha[ind+0,len+1]∧.≥syn
          :EndIf
          :If good>0∊(⊂ind+⍳len)⌷mask,0
              np←mo.Groups.Count
              r←r,⊂↑mo.Groups[⍳np].(Index Length)
          :EndIf
          mo←mo.NextMatch
      :EndWhile
     
     end:
      r←↑r ⍝ disclose results
    ∇

    if←/⍨

    ∇ r←a PIO b
     ⍝ Progressive IOTA
      r←(b/a)+(⍳+/b)-b/¯1↓0,+\b
    ∇

    ∇ REDEFINE ns;t;alls
      alls←(ns.⎕NL-2)~'Arguments' 'SwD' ⍝ all switches names
      ⍎t,'←ns.(',(t←⍕alls),')'          ⍝ redefine all vars locally
      Strings←ns.Arguments              ⍝ the strings to search
      :If EXECUTE
          Strings←,{⊂⍣(1=≡⍵)⊢⍵},CurSpace⍎⍕Strings     ⍝ argument is an expression
      :EndIf
    ∇

    ∇ r←p REPL t;k;b;Tl;Fp;Fl;∆;rstr
     ⍝ Replace Text at Positions using global Rstr
      Tl←≢¨rstr←REPLX t p Rstr ⋄ ∆←Tl-Fl←p[;0;1]
      k←(Fp,⍴t)-0,Fl+Fp←p[;0;0]
      b←(2×⍴k)⍴1 0
      t←((,k,[0.1]Fl,0)/b)/t ⋄ t←(b←(,k,[0.1]Tl,0)/b)\t
      t[(~b)/⍳⍴b]←⊃,/rstr ⋄ Fl←p[;0;0]←p[;0;0]+¯1↓0,+\∆ ⋄ p[;0;1]←Tl
      Fp←b/⍳⍴b←CR=t ⋄ p[;0;4]←Fl-b+×b←(0,Fp)[p[;0;3]←+/Fl∘.>Fp] ⋄ r←p t
    ∇

    ∇ r←REPLX(string pos pats);N;new;strs;b;ip;i;s;keep
     ⍝ Replace RegeX patterns program
     ⍝ Replacement strings can contain \0 to \9 or \A to \Z
      r←pats[ip←pos[;0;2]]
      →0↓⍨REPLACE∧PATTERN>0
      r←⍳0 ⋄ N←(1⌷⍴pos)↑⎕D,⎕A ⍝ the only possible replacements
      :For i :In ⍳⍴ip
         ⍝ Find the \d positions, account for \\ to escape \
          b←(s>¯1↓0,s)PartUS s←'\'=new←⊃pats[ip[i]] ⍝ the good (odd) \s
          keep←s≤b ⍝ remove the even \s immediately following another (odd) \
          keep>←1⌽b←(new∊N)∧¯1⌽b
          :If ∨/b
              strs←pos[i;;1]↑¨pos[i;;0]↓¨⊂string ⍝ the replacement strings
              (b/new)←strs[N⍳b/new]
          :EndIf
          r←r,,/keep/new
      :EndFor
    ∇

    RTB←{⍵↓⍨-⊥⍨' '=⍵} ⍝ Remove Trailing Blanks

    ∇ r←typ SEARCH t;h;m;l;n;S;i;s;w;ll1;nl;R;r0;⎕TRAP;syn;L1;cr;lim
      r←r0←0 1 5⍴0
     ⍝ Are we excluding BOTH locals and globals?
      →0/⍨∧/L1←¯2↑EXCLUDE∧(∨/'FS'∊typ)∨'      ∇'≡7↑t ⍝ for programs only
      →(⍴t)↓i←0 ⋄ nl←cr/⍳⍴cr←t∊CR,NL
      n←⍴Strings ⋄ r←0 1 3⍴0
      :If ∨/L1 ⍝ do we need to exclude/include those in program headers?
          ll1←(⍴t)↑((1↑nl,⍴t)⌊t⍳'⍝')⍴1 ⋄ lim←2 1⍴0,⍴t
      :AndIf 'S'∊typ ⍝ account for scripted form here
          s←s⍱cr PartOS(t='⍝')>s←cr PartUS t=''''
          m←s∧t∊'{}' ⋄ w←0≠+\¯1*'}'=m/t ⋄ s>←¯1⌽≠\m\w≠¯1↓0,w ⍝ mask out D-fns
         ⍝ Any line that starts with a ∇ is now a TradPgm header or footer
          ll1←s∧cr PartOS(¯1⌽cr PartAS cr∨' '=t)∧'∇'=t
          lim←⍉((0.5×⍴m),2)⍴m←nl/⍨1∊¨cr⊂ll1 ⍝ each program's line limits
      :EndIf
      m←0 HIDE t
      ⎕TRAP←DEBUG↓⊂0 'C' '⎕←⎕DM ⋄ →⍴r←r0'
      :If syn←S←∨/SYNTACTIC ⋄ S←~0,(t∊Cset),0 ⋄ :EndIf
      :If INSENSITIVE ⋄ t←LowerCase t ⋄ :EndIf
     ⍝ OK, we're set: search each string in turn in the present code
     l1:→(w←⍴s←i⊃Strings)↓e
      :If 0<PATTERN
          R←⍙Find s t S m SYNTACTIC ⋄ h←R[;0;0] ⋄ w←R[;0;1]
      :Else
          h←{⍵/⍳⍴⍵}s⍷t
          :If syn
              h←(S[h∘.+0,1+w]∧.≥SYNTACTIC)/h
          :EndIf
          h←(∧/m[h∘.+⍳w])/h ⍝ all valid?
      :EndIf
     ⍝ If any hit in the first line we skip this item if so desired
      :If 1∊L1
       ⍝ Find which program each hit belongs to
          l←ll1[h]/s←(1+⍳1↓⍴lim)+.×≠⌿lim∘.<h   ⍝ which programs hit in header
          h←(L1[1]=s∊l)/h
      :EndIf
      →(⍴h)↓e
      :If REPLACE>PATTERN ⍝ in replace mode update mask as we go along
          :While ~∧/s←≤\h≥¯1↓0,w+h ⋄ h←s/h ⋄ :EndWhile ⍝ remove overlaps
      :EndIf
      s←⍴h ⋄ w←s⍴w
      :If REPLACE ⋄ m[h PIO w]←0 ⋄ :EndIf  ⍝ update mask (w may be variable)
      :If 0=PATTERN ⋄ R←(s,1 2)⍴h,[0.1]w   ⍝ add length
      :ElseIf 0=i ⋄ r←R,i ⋄ →e ⍝ 1st time set r
      :EndIf
     l8:r←r{a←a⌈m←0,1↓(a←⍴⍺)⌈w←⍴⍵ ⋄ (a↑⍺)⍪(m⌈w)↑⍵}R,i
     e:→(n>i←i+h←R←1+REPLACE)/l1 ⋄ r←r[⍋r;;] ⍝ order result
      l←+⌿nl∘.<s←r[;;0] ⋄ r←(r,l),s-w+×w←(0,nl)[l] ⍝ lines #,  position in it
      r[;;2]←r[;;2]÷R ⍝ adjust string #
    ∇

    ∇ r←nam TYPE val;t;s
     ⍝ Return a 1 letter defining the type of the value
      r←'SI'⌷⍨{0::1 ⋄ 0{⍺}⎕SRC ⍵}val   ⍝ scripted? I=Invalid
      →0/⍨r≠'I'
      →(9∊↑{⍺.⎕NC⍕⍵}/nam)⍴l10×RECURSIVE ⍝ ←global
      t←≡val ⋄ →(s←+/⍴⍴val)⍴⍙ ⋄ →(0≡t)↓⍙ ⋄ →⍙×'[Null]'≢⍕val
     l10:→(∨/'( '∊1⍴⍕val)⍴0
      :Trap 6 ⋄ r←(⍎/nam).⎕WG'type' ⋄ :Else ⋄ r←(⍎/nam).Type ⋄ :EndTrap
      r←'SSPII'['Cla' 'Int' 'Nam' 'Net'⍳⊂3↑r] ⋄ →0
     ⍙:r←'E' ⋄ →((s=0)∧1≡t)/0 ⍝ ⎕OR?
      r←'ECCN'[t←6 0 2⍳10|⎕DR 1/val] ⋄ →t↓0 ⋄ →(3>⍴⍴val)⍴0 ⋄ r←'E'
    ∇

:Endnamespace ⍝ wsloc  $Revision: 1833 $
