﻿:Class Fsm

⍝ -- Finite State Machine class

⍝ This class takes a list of possible inputs and a table of states transitions
⍝ and produces an instance of a machine it represents.

⍝ It is modeled on J's fsm function (;:)
⍝ It takes a minimum of 3 things to initialize the instance:
⍝ 1. the state transition table: a NS×NI array.
⍝    There is a value for each combination of possible state and possible input which represents the 
⍝    next state for input I when in state S at position A[S;I]. An invalid state is represented by ¯1.
⍝ 2. the input groups: either a list of numbers in the range 0-255 or a list of common elements.
⍝    If not specified then it is defaulted as ⍳256 effectively making the input the mapping.
⍝ 3. A possible output code whose meaning is described below.

⍝ Some values are important as the machine processes the input:
⍝ - the interation index (II): which input is being processed
⍝ - marker index (MI): a pointer in the input list
⍝ - current state (CS)
⍝ - current mapped input (CM) index
⍝ - next state (NS)
⍝ - output code (OC)

⍝ By default II runs over ⍳⍴input. MI is set to ¯1 and is reset according to the
⍝ output codes in the state table. CS starts at ⍳1 and is also reset according
⍝ to the state table. Non-default values can be specified when using an instance.
⍝ The output code has the following values:
⍝    m: reset marker to the iteration index
⍝    M: same as 1 + produce some result
⍝    E: reset marker to ¯1 + produce some result
⍝    s: stop
⍝ By default the result is ⊂input[MI+⍳II-MI] but this can be changed by supplying
⍝ a function which accepts II,MI,CS,CM,NS,OC as right argument. For example, a tracing
⍝ function could be {⊂⍵} which simply returns its boxed argument.

⍝ When an instance is created the parameters can be checked for consistency.

⍝ When an instance is used it needs an input sequence and possibly some initial state values.
⍝ 1. the function code

    ⎕io←0 ⋄ ⎕ml←1

    isChar←{∨/0 2∊10|⎕dr ⍵}
    if←/⍨

    ∇ init2(i m)
      :Access public
      :Implements constructor
      init i m 0
    ∇

    ∇ init3(i m o)
      :Access public
      :Implements constructor
      init i m o
    ∇

    ∇ init arg;gtt;sh;set;t;illegal;tmp
    ⍝ Prepare the instance
     
      (Inputs Movements Output)←arg
      'Movements must be a matrix'⎕SIGNAL 11 if 2≠⍴⍴Movements
      doOutput←Output≢0
     ⍝ By default we produce an enclosed element
      fnOutput←{⊂⍺[b+⍳-/(a b)←2↑⍵]}
      :If (1≡≡t)∧0∊⍴⍴t←Output
          'tmp'⎕NS''
          fnOutput←tmp⍎tmp.⎕FX Output
          Output←Movements{⍵}¨'E' ⍝ make a call to the fn at every move
      :EndIf
      Col←{Inputs⍳⍵} ⍝⋄ Row←{⍵}
     ⍝ If sets of char vectors they are translated into a 256 integers vector
      :If 326∊⎕DR set←Inputs←,Inputs ⋄ :AndIf ∧/(set←∊Inputs)∊⎕AV  ⍝ VTVs?
          t←(∊sh⍴¨⍳⍴sh),⍴sh←⍴¨Inputs←,¨Inputs
          Inputs←t[set⍳⎕AV]  ⍝ turn characters into integers
          Col←{Inputs[⎕AV⍳⍵]}
      :EndIf
      'Inputs not unique'⎕SIGNAL 11 if set≢∪set
      States←⍳1↑⍴Movements
      :If Translate←isChar Movements
      :OrIf (⍳⍴t)≢t[⍋t←∪,Movements]
         ⍝ The 1st column is the states, the rest is the transitions
          illegal←¯1=t←0 1↓Movements
          States←Movements[;0] ⋄ Movements←illegal{(⍵×~⍺)-⍺}States⍳t
      :EndIf
      'Invalid State'⎕SIGNAL 11 if∨/{(⍵<¯1)∨⍵≥⍴States}∪,Movements
      CurrentState←0
     
      ⎕DF'FSM ',(⍕⍴Inputs),doOutput/'+ Actions'
    ∇

    ∇ transit←{showoutput}Input arg;actions;act;in;new;default;col;drop;marker
      :Access public
      (actions default)←2↑(⊂⍣(1∊≡arg)⌷arg),0
      (drop marker CurrentState)←3↑default,(⍴,default)↓0 ¯1 0
      :If 0=⎕NC'showoutput' ⋄ showoutput←0 ⋄ :EndIf
      transit←⍬
      :For in :In drop↓⍳⍴actions
          PreviousState←CurrentState
          new←Movements[CurrentState;col←Col in⌷actions]
          :If doOutput
              :Select act←Output[CurrentState;col]
              :Case 'm' ⋄ marker←in
              :Case 'M' ⋄ transit,←arg fnOutput in marker CurrentState col new act ⋄ marker←in
              :Case 'E' ⋄ transit,←arg fnOutput in marker CurrentState col new act ⋄ marker←¯1
              :EndSelect
          :Else
              transit,←new
          :EndIf
         ⍝ Reset the machine on invalid or final input
          CurrentState←0⌈new
          :If showoutput ⋄ PreviousState,'→',new ⋄ :EndIf
      :EndFor
     
      :If Translate>doOutput ⋄ transit←States[transit] ⋄ :EndIf
    ∇

⍝ There are many possible examples. 4 are shown here.
⍝ 1. find text within a string using quotes
⍝ 2. find text within s string using either quotes or double quotes
⍝ 3. find APL identifiers in a statement
⍝ 4. play the NIM game

⍝ 1. An example is to find where text resides in a statement.
⍝    For example, in the statement
⍝ APL←'string',(condition)/'another string' {fn} var
⍝ there are 2 strings and a finite state machine to detect the quotes could be defined as
⍝   fq←⎕NEW Fsm ('''' (2 2⍴1 0 0 1)) ⍝ 0=string, 1=quoted text
⍝ and doing "fq.Input text" is the same as "≠\''''=text".

⍝ This is not quite right as some quotes are considered text so here's
⍝ 1b: this fixes the quote being part of the strings:
⍝   fq←⎕NEW Fsm ('''' ('s12q',4 2⍴'1s2qqs2q')) ⍝ 1=1st quote, 2=2nd quote
⍝   fq.Input txt←⍞
⍝ 'I can''t' he said
⍝ 1qqqqq2qq2ssssssss
⍝ and
⍝   's'≠fq.Input txt
⍝ yields the proper answer

⍝ 2. A more interesting problem is when ' and " are used a la APL+Win to denote
⍝    strings. In that case we need (n=not quoted, s=single quote text, d=double quote text)
⍝   fq←⎕NEW Fsm ('''"' ('n12s34d',7 3⍴'13n 2ss s3n 2ss  d4d 1dn d4d'~' ') )
⍝   fq.Input txt←⍞
⍝ "I can't" said he 'all of a "sudden"'!
⍝ 3ddddddd4nnnnnnnnn1sssssssssssssssss2n
⍝ and
⍝   'n'≠fq.Input txt
⍝ yields the proper answer

⍝ 3. An APL identifier is a name starting with ⎕ followed by a few alpha characters
⍝    or an alpha character followed by 0 or more alphanumeric characters.
⍝    It looks like this:
⍝   chars←'⎕' '_abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'  '0123456789'
⍝   fq←⎕new Fsm (chars ('nQ⎕⍺',4 4⍴'Q⍺nn Q⎕nn Q⎕nn Q⍺⍺n'~' '))
⍝   fq.Input ⍞
⍝ ⎕io⎕9⎕nn-nam⎕io1n1
⍝ Q⎕⎕QnQ⎕⎕n⍺⍺⍺Q⎕⎕n⍺⍺

⍝    Unfortunately this is insufficient information to determine where the identifiers are.
⍝ We need to supply output codes:
⍝   fq←⎕NEW Fsm (chars ('nQ⎕⍺',4 4⍴'Q⍺nn Q⎕nn Q⎕nn Q⍺⍺n'~' ')(4 4⍴'mm00m000M00EM00E'))
⍝   fq.Input '⎕io⎕9⎕nn-nam⎕io1n1'
⍝  ⎕io  ⎕nn  nam

⍝ 4. The game of NIM with 3 rows of 1, 2 and 3 sticks where the aim
⍝ is to grab the last stick and any number of sticks in a row can be grabbed at
⍝ a single time (but on no more than 1 row),
⍝ can be represented by a machine like this:
⍝ (fresh game of 1, 2 and 3 sticks, the initial state)
⍝ there are 6 possible inputs:
⍝ 1. grab 1 stick  from the 1st row
⍝ 2. grab 1 stick  from the 2nd row
⍝ 3. grab 2 sticks from the 2nd row
⍝ 4. grab 1 stick  from the 3rd row
⍝ 5. grab 2 sticks from the 3rd row
⍝ 6. grab 3 sticks from the 3rd row

⍝ There are 24 possible states (2×3×4 combinations of sticks on each row)
⍝ but only a subset leads to a winning (final) state (000). Those are
⍝ 123, 022, 011, 101, 110, 0 (final)

⍝ The inputs are: 11, 12, 22, 13, 23, 33
⍝ The states are: 123, 022, 011, 101, 110, 000
⍝ The movements are (action/new state):
⍝       11     12     22    13     23     33
⍝ 123  13/22 33/110 23/101 11/22 22/101 12/110
⍝  22    -   13/11  23/0   12/11 22/0     -
⍝ 110  12/0  11/0     -      -     -      -
⍝ 101  13/0    -      -    11/0    -      -
⍝  11    -   13/0     -    12/0    -      -

⍝ To construct a Finite State Machine to represent that game you could use here
⍝ A←5 6⍴22 110 101 22 101 110,¯1 11 123 11 123 ¯1,123 123 ¯1 ¯1 ¯1 ¯1,123 ¯1 ¯1 123 ¯1 ¯1,¯1 123 ¯1 123 ¯1 ¯1
⍝ ⎕new Fsm ( (11 12 22 13 23 33) (123 22 11 101 110 ,A))

⍝ If output is desired we can use
⍝ ⎕FX'r←a F b' 'r←⊂(5 6⍴13 33 23 11 22 12,0 13 23 12 22 0,12 11 0 0 0 0,13 0 0 11 0 0,0 13 0 12 0 0){''I take'',n,''on row'',1↓n r←0 10⊤⍵[2 3]⌷⍺⍺}b'
⍝ ⎕new Fsm ( (11 12 22 13 23 33) (123 22 110 101 11,A) (⎕OR'F') )

    ∇ tests;text;a;fq;chars;A;F
      :Access public shared
     ⍝ Test the above comments
      a←{a←÷∧/⍵}
      fq←⎕NEW Fsm(''''(2 2⍴1 0 0 1)) ⍝ 0=string, 1=quoted text
      a(fq.Input text)≡≠\''''=text←'dsa''asd''''ewq''qwe'
      fq←⎕NEW Fsm(''''('s12q',4 2⍴'1s2qqs2q')) ⍝ 1=1st quote, 2=2nd quote
      a'1qqqqq2qq2ssssssss'≡fq.Input'''I can''''t'' he said'
      fq←⎕NEW Fsm('''"'('n12s34d',7 3⍴'13n  2ss s3n 2ss  d4d 1dn d4d'~' '))
      a'3ddddddd4nnnnnnnnn1sssssssssssssssss2n'≡fq.Input'"I can''t" said he ''all of a "sudden"''!'
      chars←'⎕' '_abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ' '0123456789'
      fq←⎕NEW Fsm(chars('nQ⎕⍺',4 4⍴'Q⍺nn Q⎕nn Q⎕nn Q⍺⍺n'~' '))
      a'Q⎕⎕QnQ⎕⎕n⍺⍺⍺Q⎕⎕n⍺⍺'≡fq.Input'⎕io⎕9⎕nn-nam⎕io1n1'
      fq←⎕NEW Fsm(chars('nQ⎕⍺',4 4⍴'Q⍺nn Q⎕nn Q⎕nn Q⍺⍺n'~' ')(4 4⍴'mm00m000M00EM00E'))
      a'⎕io' '⎕nn' 'nam'≡fq.Input'⎕io⎕9⎕nn-nam⎕io1n1'
      A←5 6⍴{⍵+123×0=⍵}22 110 101 22 101 110,¯1 11 0 11 0 ¯1,0 0 ¯1 ¯1 ¯1 ¯1,0 ¯1 ¯1 0 ¯1 ¯1,¯1 0 ¯1 0 ¯1 ¯1
      fq←⎕NEW Fsm((11 12 22 13 23 33)(123 22 110 101 11,A))
      a 1 4≡fq.Input 11 12
      ⎕FX'r←a F b' 'r←⊂(5 6⍴13 33 23 11 22 12,0 13 23 12 22 0,12 11 0 0 0 0,13 0 0 11 0 0,0 13 0 12 0 0){''I take'',n,''on row'',1↓n r←0 10⊤⍵[2 3]⌷⍺⍺}b'
      fq←⎕NEW Fsm((11 12 22 13 23 33)(123 22 110 101 11,A)(⎕OR'F'))
      fq.Input 11 12
    ∇

    ∇ NimExample;A;O;s;i;t
      :Access public shared
      A←A+123×0=A←5 6⍴22 110 101 22 101 110,¯1 11 0 11 0 ¯1,0 0 ¯1 ¯1 ¯1 ¯1,0 ¯1 ¯1 0 ¯1 ¯1,¯1 0 ¯1 0 ¯1 ¯1,6/0
      t←6 6⍴13 33 23 11 22 12,0 13 23 12 22 0,12 11 0 0 0 0,13 0 0 11 0 0,0 13 0 12 0 0,6⍴¯1
      O←{⍕'I take'⍺'from'⍵}⌿0 10⊤t
      #.F←⎕NEW Fsm((i←11 12 22 13 23 33)((s←123 22 11 101 110 0),A))
      'Table is:' ⋄ (' ',s),i⍪t{⍺=0:'?' ⋄ (⍕⍺),'→',⍕⍵}¨A
      'Now try F.Input n-row(s) (e.g. 11 12); the moves wil be shown and the new states returned'
    ∇


⍝ Here are some utilities to assert the initial code is valid

    assert←{a←÷∧/,⍵}

    ∇ la smcheck ra;f;s;m;ijr;q;i;j;r
    ⍝ Check the argument to a FSM
      :Access public
      assert 326∊⎕DR la      ⍝ la must be an enclosure
      assert(⍴la)∊2 3 4
      (f s m ijr)←4↑la,0 0
      assert(0∊⍴⍴f)∧f∊⍳6
      assert(3∊⍴⍴s)∧2∊¯1↑⍴s  ⍝ 3D, 2 col
      assert(0≤s)∧s≡⌊s       ⍝ all ints ≥0
      assert(1↑⍴s)>⌈/s[;;0]  ⍝ states all valid
      assert s[;;1]∊⍳7
      assert 1∊⍴⍴m           ⍝ vector
      q←1⌷⍴s
      :If 0∊⍴m
          assert(1∊⍴⍴ra)∧ra≡⌊ra  ⍝ ra used as is
          assert q>ra
      :ElseIf ∨/1 3∊10|⎕DR m ⍝ numeric
          assert q>m
          assert(⍴m)≡⍴⎕AV    ⍝ m is mapping on ⎕AV
          assert m∊⍳⍴⎕AV     ⍝ all within range
      :Else
          assert 326≡⎕DR m   ⍝ must be enclosed
          assert q>1↑⍴m
          assert(⍴⍴⊃,/m)∊0 1+⍴⍴ra
      :EndIf
      :If 0<s←⍴,ijr
          assert 3=s
          assert ijr≡⌊ijr
          (i j r)←ijr
          assert(0≤i)∧i<⍴ra
          assert(¯1=j)∨(0≤j)∧j<i
          assert(0≤r)∧r<q
     
      :EndIf
    ∇


:EndClass ⍝ Fsm  $Revision: 739 $ 