﻿:Namespace fileUtils ⍝ V1.09
⍝ This namespace contains various file utilities. Some require .Net.
⍝ 2015 05 19 Adam: dywftt domain note, replExpr properly returns number of changes (not files), fileRename properly unties
⍝ 2015 06 22 Adam: Fixed confusing comment typo
⍝ 2015 10 31 DanB: filesIn was returning more files than necessary and removed folders from result
⍝ 2015 12 13 DanB: changed dywtss to FileTStoQTS

    ⎕io←⎕ml←1

    BigEndian←⊣/83 ⎕dr 256

    ∇ Chars←ReadFile name;nid;signature;nums;sz;b
     ⍝ Read ANSI or Unicode character file
      sz←⎕NSIZE nid←name ⎕NTIE 0
      signature←⎕NREAD nid 83 3 0
      :If signature≡¯17 ¯69 ¯65 ⍝ UTF-8?
          nums←⎕NREAD nid 83 sz
          Chars←'UTF-8'⎕UCS 256|nums ⍝ Signed ints
      :ElseIf ∨/b←(2↑signature)∧.=2 2⍴¯1 ¯2 ¯2 ⍝ Unicode (UTF-16)
          Chars←{,⌽(2,⍨2÷⍨⍴⍵)⍴⍵}⍣(⊣/b)⎕NREAD nid 83 sz 2
          Chars←'UTF-16'⎕UCS(2*16)|163 ⎕DR Chars
      :Else ⍝ ANSI or UTF-8
          Chars←{11::⎕UCS ⍵ ⋄ 'UTF-8'⎕UCS ⍵}256|⎕NREAD nid 83 sz 0
      :EndIf
      ⎕NUNTIE nid
    ∇

    ∇ {format}WriteFile(name chars);nid;signature;nums;typ
      ⍝ Write ANSI or Unicode character file
      ⍝ format = ANSI, UTF-8 or UTF-16
      ⎕SIGNAL(×10|typ←⎕DR chars)/⊂('EN' 11)('Message' 'Does not work in Classic APL')
      :If 0=⎕NC'format'
          format←'ANSI' 'ANSI' 'UTF-16'⊃⍨80 82⍳typ
      :Else
          format←'ANSI' 'UTF-8' 'UTF-16' 'UTF-8' 'UTF-16' '?'⊃⍨'ANSI' 'UTF8' 'UTF16' 'UTF-8' 'UTF-16'⍳⊂format
          ⎕SIGNAL(format≡'?')/⊂('EN' 11)('Message' 'Invalid file translation format')
      :EndIf
      :Trap 22
          nid←name ⎕NCREATE 0
      :Else
          name ⎕NERASE name ⎕NTIE 0
          nid←name ⎕NCREATE 0
      :EndTrap
     
      :Select format
      :Case 'ANSI'
          chars ⎕NAPPEND nid 80
      :Case 'UTF-8'
          ¯17 ¯69 ¯65 ⎕NAPPEND nid 83
          nums←⎕UCS'UTF-8'⎕UCS chars
          nums ⎕NAPPEND nid 80
      :Else
          (BigEndian⌽¯1 ¯2)⎕NAPPEND nid 83
          {}⎕DR nums←{⍵-(2*16)×⍵≥2*15}'UTF-16'⎕UCS chars ⍝ make sure it is type 163
        ⍝ We write as a stream of bytes to avoid little endian problems
          (83 ⎕DR nums)⎕NAPPEND nid 83
      :EndSelect
      ⎕NUNTIE nid
    ∇

    ∇ yes←Exists path;GFA ⍝ for Windows only
    ⍝ Is the argument the name of an existing file or folder?
      :Trap 2
          yes←⎕NEXISTS path
      :Else
          :If ⎕SE.SALTUtils.WIN
              'GFA'⎕NA'U4 kernel32.C32|GetFileAttributes* <0T '
              yes←(¯1+2*32)≢GFA⊂path
          :Else
              yes←(,'0')≡⎕IO⊃⎕SH'test -f ',path,'; echo $?'
          :EndIf
      :EndTrap
    ∇


    ∇ ts←FileTStoQTS ts;⎕IO;shape;y;hms
    ⍝ Convert ⎕FRDCI type file timestamp(s) to ⎕TS format
    ⍝ Right argument: ⎕FRDCI-type timestamp(s) of any shape
    ⍝ Result: ⎕TS type timestamp(s) with same shape and 7 columns
      ⎕IO←0 ⋄ shape←⍴ts
      hms←⍉1↓y←⌊0 24 60 60 60⊤,ts ⋄ y←25568+y[0;]
      hms[;3]←⌊0.5+100×hms[;3]÷6
      ts←(shape,7)⍴(0 ¯1↓↑{2 ⎕NQ'.' 'IDNToDate'⍵}¨y),hms
    ∇

    Split←{~1=≡⍵:⍵ ⋄ ⍺←' ' ⋄ ⎕ML←3 ⋄ (~⍵∊⍺)⊂⍵} ⍝ don't touch enclosed

    DefaultExtensions← Split 'dyalog asmx aspx asax apl'

    Options←{('IC'(∨/'Ii'∊⍵)) ('Mode'('DML'[1⍳⍨∨/2 2⍴'DdMm'∊⍵]))}

    ∇ msg←files showExpr arg;adir;file;oname;regex;uf;filedir;hits;⍙rec;⍙opt;⍙types;fs;count;report;ws;tmp;H;noNS
    ⍝ This fn will show where Expressions (arg) occur in files
     
    ⍝ The left argument to this fn is a string representing a file or a directory
    ⍝ It can include the following switches:
    ⍝  -types=extension   only the files with that extension are scanned (default .dyalog .asmx .aspx .asax and .apl )
    ⍝ If the file type is DWS the file is assumed to be a workspace and the search is done in APL code.
    ⍝  -options=IDML      search Insensitive, mode Document, Mixed or Line (default)
    ⍝  -rec               to search recursively
    ⍝ The right arg is the string to search, it is a full fledge regular expression.
    ⍝ Ex: 'C:\Dyalog101\samples\ -types=txt' showExpr '⎕\w+←\d+' ⍝ look for a ⎕var←number in txt files
     
      msg←'* nothing found'
     
      :If ~(10|⎕DR arg)∊0 2 ⍝ if the arg is enclosed then
          (arg ⍙types ⍙rec ⍙opt)←arg,(⍴,arg)↓0 ⍬ 0 ⍬ ⋄ filedir←files
      :Else
          fs←(⎕NEW ⎕SE.Parser,⊂'-types= -rec -options∊iIMmDdLl' 'upper').Parse files
          →0⍴⍨' '∧.=arg,filedir←1↓∊' ',¨fs.Arguments
          (⍙rec ⍙opt ⍙types)←fs.(REC OPTIONS TYPES)
      :EndIf
     
    ⍝ If the type is DWS it must be the only one
      ws←∨/⍙types∘≡¨'dws' 'DWS'
     
    ⍝ Validate the argument
      :If 0∊⍴oname←(⍙types~0)filesIn filedir,⍙rec⍴'↓'
          msg←'* 0 file found in ',filedir ⋄ →0
      :EndIf
      count←0 ⋄ noNS←((⍕⎕THIS),'.tmp')⎕R'#'
     ⍝ The report fn returns a line #, the line that matched and a boolean showing the match of the highest group unless
     ⍝ the entire match is of length 0 (possible if \K was used)
      report←{0∊1↑L←⍵.Lengths:0
          bl←⍴ln←'[',(⍕num←⍵.BlockNum),'] ' ⋄ sp←0↑⍨⍴ln←ln,⍵.Block ⋄ sp[(bl+¯1↑⍵.Offsets)+⍳¯1↑L]←~⎕IO←0
          num ln sp}
     
      regex←arg ⎕S report⍠(Options ⍙opt)
      :For file :In oname
          :Trap 0
              :If ws
                  ⎕EX'tmp' ⋄ 'tmp'⎕NS''
                  :If ⎕SE.SALTUtils.DEBUG>0 ⋄ ⎕←file
                      fs←2 ⎕NQ'.' 'setdflags' 256
                      tmp.⎕CY file
                      2 ⎕NQ'.' 'wscheck'
                      2 ⎕NQ'.' 'setdflags'fs
                  :Else
                      tmp.⎕CY file
                  :EndIf
                  tmp←⎕SE.UCMD'locate "',arg,'" -recursive=tmp -return -nomessage'
                  :If ~0∊⍴tmp
                      count+←+/2⊃⎕VFI∊'\((\d+ )found\)'⎕S'\1'⊢tmp
                      ⎕←'' ⋄ ⎕←'>>> WorkSpace  ',file ⋄ ⎕←noNS tmp
                  :EndIf
              :Else
                  uf←ReadFile file
                  :If 0≠count+←⍴hits←(regex uf)~0
                      fs←↑{(⊃⍺),[0.1]'∧'\⍨↑∨/⍵}⌸/1↓↓⍉↑hits
                      ⎕←'' ⋄ ⎕←0⍴⎕←file ⋄ ⎕←fs
                  :EndIf
              :EndIf
          :Else
              ⎕←'*** Unable to process file "',file,'" :',{⍵.EM,(1<⍴m)/' (',m←⍵.Message,')'}⎕DMX
          :EndTrap
      :EndFor
      msg←⍕count'matches found'
    ∇

    Eis←{⊂⍣(1=≡⍵)⊢⍵}

    ∇ r←{types}filesIn name;f;list;rec;ne;ext
    ⍝ Return file names in folder without using .Net
      f←-rec←'↓'∊¯1↑name
      list←⎕SE.SALT.List(f↓name),' -full=2 -extension -raw ',rec/' -recursive'
      list←(0=∊⍴¨list[;1])⌿list[;2] ⍝ keep full file path only
      :If 0=⎕NC'types' ⋄ :OrIf ⍬≡types
          types←DefaultExtensions ⋄ ne←0
      :ElseIf ~ne←''≡types       ⍝ '' means files without an extension
          types←' ,'Split types  ⍝ strings or VTVs go here
      :EndIf
      :If ne ⍝ no extension?
          r←list/⍨{~1∊d←'.'=⍵:1 ⋄ ~1∊d>⌽∨\⌽⍵∊'/\'}¨list
      :Else
         ⍝ Limited regex: ? means 1 char, * means many, [xyz] is allowed
          ext←{'^.*\.','\?' '\*' '[{}()?*.+\\]'⎕R'.' '.*' '\\\0'⊢⍵,'$'}¨types ⍝ turn limited regex into full regex
          r←list[⎕IO+ext ⎕S 2⍠⎕SE.SALTUtils.WIN⊢list]
      :EndIf
    ∇

    ∇ r←{types}DNetfileInfo name;⎕USING;f;tree
    ⍝ Return file information   - Windows only
      :If 0=⎕NC'types' ⋄ :OrIf 0∊⍴types
          types←'*.'∘,¨DefaultExtensions
      :Else
          types←{(1↓¨⍵⊂⍨⍵=' ')~⊂''}' ',types
          types←{'*'∊⍵:⍵ ⋄ '.'=1↑⍵:'*',⍵ ⋄ '*.',⍵}¨types
      :EndIf
      ⎕USING←'System.IO'
      tree←'↓'∊name ⋄ name~←'↓'
      r←⍬ ⋄ f←⎕NEW FileInfo,⊂⊂name
     
      :If f.Exists ⋄ r←f
      :ElseIf tree ⋄ r←types treefileInfo name
      :ElseIf (f←⎕NEW DirectoryInfo,⊂⊂name).Exists
          r←⊃,/f.GetFiles¨⊂¨types
      :EndIf
    ∇

    ∇ r←{types}treefileInfo name;⎕USING;f
    ⍝ Return file information in form of tree - Windows only
      ⎕USING←'System.IO'
      r←⍬ ⋄ f←name
      :If 9≠⎕NC'name' ⋄ f←⎕NEW DirectoryInfo,⊂⊂name ⋄ :EndIf
      :If f.Exists
          :If 0=⎕NC'types' ⋄ :OrIf 0∊⍴types ⋄ types←'*.'∘,¨DefaultExtensions ⋄ :EndIf
          r←⊃,/f.GetFiles¨⊂¨{1≡≡⍵:⊂⍵ ⋄ ⍵}types
          :If ×⍴f←f.GetDirectories⊂,'*'
              r←r,⊃,/types∘treefileInfo¨f
          :EndIf
      :EndIf
    ∇

    ∇ files←{ext}recFLib path;name;dirs;n;⎕USING
    ⍝ Recursively find all component files starting at path - Windows only
      ⎕USING←'System.IO'
      :If 9≠⎕NC'path' ⋄ path←⎕NEW DirectoryInfo,⊂⊂path ⋄ :EndIf
      files←⊂⎕FLIB path.FullName
      :If path.Exists
      :AndIf ~0∊⍴dirs←path.GetDirectories⊂,'*'
          files,←recFLib¨dirs
      :EndIf
      files/⍨←(∧\n)⍱⌽∧\⌽n←' '∧.=files←⍕⍪files ⍝ trim
      :If 2=⎕NC'ext' ⋄ :AndIf ext
          files←n⌽(files⌽⍨-n←+/∧\⌽' '=files),(files∧.≠'.')⍀1 4⍴'.DCF'
      :EndIf
    ∇

    ∇ msg←files replExpr arg;adir;file;oname;regex;uf;filedir;⍙rec;⍙opt;⍙types;re;wh;new;changes;ps;fs;⍙bu;occur;work
    ⍝ This fn will replace an expression by another in files
     
    ⍝ The left argument to this fn is a string representing a file or a directory
    ⍝ It can include the following switches:
    ⍝  -types=extension   only the files with that extension are scanned (default .dyalog .asmx .aspx .asax and .apl )
    ⍝  -options=IDM       to search Insensitive, Document mode, Mixed mode
    ⍝  -rec               to search recursively
    ⍝  -backup[=.bu]      take a backup of the original file (using .bu as default file extension)
     
    ⍝ The right arg is the string to search, a full fledged regular expression,
    ⍝ followed by another expression to perform replacement. Both are separated by a space.
    ⍝ If the expression include space they must be surrounded by quotes.
    ⍝ Ex: '\temp\ -types=PHP' replExpr '(\w*),(\w*) ''\2 \1''' ⍝ change word1,word2 into word2 word1
     
    ⍝ You should take backup copies of your files with -backup when using this function on them.
    ⍝ The changes made are irreversible!
     
      msg←'* 0 changes made'
     
      :If ~(10|⎕DR arg)∊0 2
          (wh re ⍙types ⍙rec ⍙opt ⍙bu)←arg,(⍴,arg)↓0 0 ⍬ 0 ⍬ 0 ⋄ filedir←files
      :Else
          ps←⎕NEW ⎕SE.Parser,⊂'-types= -rec -options∊iIdDMmLl -backup[=]' 'upper'
          fs←ps.Parse files
          →0⍴⍨' '∧.=arg,filedir←1↓∊' ',¨fs.Arguments
          (⍙rec ⍙opt ⍙types ⍙bu)←fs.(REC OPTIONS TYPES BACKUP)
          ps←⎕NEW ⎕SE.Parser('' 'nargs=2L') ⍝ use this as fancy quote (and ") cutter
          (wh re)←(ps.Parse arg).Arguments  ⍝ use Parse to get rid of quotes (may fail if one arg starts with +)
      :EndIf
      :If 0≢⍙bu ⋄ ⍙bu←{0∊⍴⍵:'.bu' ⋄ ('.'∊⍵)↓'.',⍵}⍙bu~1 ⋄ :EndIf ⍝ default backup extension
     
    ⍝ Validate the argument
      :If 0=⍴,oname←(⍙types~0)filesIn filedir,⍙rec⍴'↓' ⋄ msg←'No file found in ',filedir ⋄ →0 ⋄ :EndIf
      changes←0
      regex←wh ⎕R re⍠(Options ⍙opt)
      occur←wh ⎕S re⍠(Options ⍙opt)
      :For file :In oname
          uf←ReadFile file
          changes+←work←≢occur uf
          :If 0≠work
              new←regex uf
              :If ⍙bu≢0
                  file fileRename file,⍙bu
              :EndIf
              'UTF-8'WriteFile file new
          :EndIf
      :EndFor
      msg←⍕changes,'changes made'
    ∇

      splitName←{f←⍵↓⍨l←-⊥⍨~⍵∊'/\'  ⍝ split filename into folder, name and extension sections
          n←nx↓⍨d←(0∊d)×-1+⊥⍨d←'.'≠nx←l↑⍵
          f n(d↑nx)}

    ∇ oldname fileRename newname;old;tie;folder;name;ext;renamefn
    ⍝ Rename file or file tied to newname.
    ⍝ Works for a regular or component file. No .Net.
      :If ∨/1 3=10|⎕DR old←oldname    ⍝ tieno acceptable
          renamefn←⎕NRENAME
          :If 0<old ⋄ renamefn←⎕FRENAME ⋄ :EndIf   ⍝ APL rename: it may fail if not exclusively tied
          newname renamefn old
      :Else
         ⍝ Rename using a string
         ⍝ Is this a component file?
          (folder name ext)←splitName oldname
          ext,←(0∊⍴ext)/'.DCF'   ⍝ no extension means component file
          tie←(folder,name,ext)⎕NTIE 0
          :If 0∊⍴1⊃splitName newname ⋄ newname←folder,newname ⋄ :EndIf
          ⎕NUNTIE newname ⎕NRENAME tie
      :EndIf
    ∇

    ∇ report←{opt}make64b file;tie;newname;newtie;nn;new;ext;list;ok;TIE
    ⍝ Transform file into 64b format.
    ⍝ file must not be tied already.
    ⍝ 'opt' is backup EXTension and "LIST only" flag
    ⍝ 'ext' allows for backing up the file with a new extension
      newtie←⍬
      report←'*** "',file,'" is tied'
      TIE←⎕FTIE ⍝ in V14 we can't access 32b files anymore
      :If 14≤⍎{⍵↑⍨⍵⍳'.'}2⊃⎕SE.SALTUtils.APLV ⋄ TIE←⎕FTIE⍠'ReadOnly' 1 ⋄ :EndIf
      :Trap 24 22
          ok←×tie←file TIE ok←0
      :Case 22
          :Trap 24 22 ⍝ try with no extension
              ok←×tie←(file,'.')TIE 0
          :Case 22
              report←'*** "',file,'" not found'
          :EndTrap
      :EndTrap
      →ok↓0
      :If 0=⎕NC'opt' ⋄ opt←0 ⋄ :EndIf
      (ext list)←2↑(⊂⍣(0 2∨.=10|⎕DR opt)+opt),0
      :If 64='S'⎕FPROPS tie
          report←'* "',file,'" is already 64b'
      :ElseIf list
          report←'"',file,'" is in 32b format'
      :Else
          report←'"',file,'" made into 64b format'
          newname←file,'.new'
          newtie←newname ⎕FCOPY tie
          :If ~0∊⍴ext←ext~0   ⍝ rename the source
              tie fileRename nn←(new←filenameOf tie),ext
              newtie fileRename new
              report,←' and backed up to "',nn,'"'
          :Else ⍝ get rid of the source
              file ⎕FERASE tie
              newtie fileRename file
          :EndIf
      :EndIf
      ⎕FUNTIE tie,newtie ⍝ Only one is actually tied
     
    ∇

    ∇ r←filenameOf tieno
    ⍝ Return the filename associated with a single tie no
      r←{(⍵↓⍨-⊥⍨' '=⍵),(~'.'∊⍵)/'.DCF'},⎕FNAMES[⎕FNUMS⍳tieno;] ⍝ bombs if not a valid tie no
    ∇

    ∇ p←filePermission cod
    ⍝ Returns a verbose list of permission from the code of
    ⍝ the access matrix of a component file
      p←' read+size tie erase append replace drop hold '
      p←p,'rename rdac+stac rdci resize fhold rdac stac copy'
      cod←⌽,(15⍴2)⊤cod
      p←1↓¨cod⌿(p=' ')⊂p
    ∇

    FN←⊃∘(//)∘⎕vfi ⍝ Fix Numbers
      FI←{bad←(0∊⍴b)≥∧/b←↑⊣/bv←⎕VFI ⍵ ⍝ at least one number, all valid
          bad∨n≢⌊n←↑⊢/bv:'not Integers'⎕SIGNAL 11
          n}

  ⍝                                    ** File Sort **
  ⍝ Sort component file on specific columns.
  ⍝ The components to be sorted are matrices whose columns are used for sorting.
  ⍝ For example, column positions 3 and 7 could be sorted in ascending order
  ⍝ while column 2 should be sorted after in descending order.
  ⍝ The specification would look like this: 'A3,7, D2'
  ⍝ tie contains the tie number followed by the component numbers if not all of them.

  ⍝ Sorting is done using the file itself with minimal use of the workspace.
  ⍝ Some growing replace may occur as a result.

    ∇ tie fileSort columns;sort;cols
     ⍝ Left argument is tie number, components to sort (⍬=all)
     ⍝ Define sorting order
      :If 83∊⎕DR columns ⋄ cols←⊂1,columns
      :Else ⋄ cols←⍕columns
          cols←cols,⍨'A'↓⍨(1↑cols)∊'AadD' ⍝ add A to the front if not specified
          cols←{('Aa'∊⍨1↑s),FI 1↓s←b\⍵/⍨b←⍵≠','}¨{(⍵∊'AadD')⊂⍵}cols ⍝ split into sorting groups
      :EndIf
      sort←{↑{⍵[⍺{1↑⍺:⍋⍵ ⋄ ⍒⍵}⍕⍣(326∊⎕DR v)⊢v←⍵[;1↓⍺-~⎕IO];]}/cols,⊂⍵}
      tie reorder sort mergesort tie
    ∇

    ∇ tie reorder order;done;next;c0;c1;oc1;oc0;cpts;io
     ⍝ Reorder file tied according to order given
      cpts←1↓tie ⋄ tie←1↑tie
      cpts,←(0∊⍴cpts)/order[io←⍋order]
      done←(⍳⍴io)=order←io
     
      :While ~∧/done
          oc0←next←done⍳0
        ⍝ Process one group
          c0←⎕FREAD tie,cpts[next]
          :Repeat
              c1←⎕FREAD tie,cpts[oc1←order[oc0]]
              c0 ⎕FREPLACE tie,cpts[oc1]
              done[oc1]←1
              c0←c1 ⋄ oc0←oc1
          :Until next=oc1
      :EndWhile
    ∇

      mergesort←{   ⍝ from RHU 2014
     ⍝ Merge sort of matrices in component file ⍵
     ⍝ Result is component numbers in sorted sequence
          ⎕IO←0
          tie←1↑⍵                                            ⍝ file tie number
          cpts←{1=≢⍵:t[0]+⍳|-/t←2⍴⎕FSIZE ⍵ ⋄ 1↓⍵}⍵           ⍝ the components to sort
          merge←⍺⍺{                                          ⍝ merge sorted components lists ⍺ and ⍵
              (⍬≡⍺)∨(⍬≡⍵):⍺,⍵                                ⍝ if either list is empty, return the other list
              p←⎕FREAD tie,1↑⍺                               ⍝ first sorted vector for ⍺
              q←⎕FREAD tie,1↑⍵                               ⍝ first sorted vector for ⍵
              i←⍋m⍳⍨⍺⍺ m←(1↑p)⍪(¯1↑p)⍪(1↑q)⍪(¯1↑q)           ⍝ control vector
              i≡0 1 2 3:(1↑⍺),⍵ ∇ 1↓⍺⊣m←p←q←0                ⍝ ()[]: p entirely precedes q
              i≡2 3 0 1:(1↑⍵),⍺ ∇ 1↓⍵⊣m←p←q←0                ⍝ [](): q entirely precedes p
              n←⌊0.5×≢m←⍺⍺ p⍪q                               ⍝ merge ("mesh") p and q
              c←(1↑⍺),(1↑⍵)                                  ⍝ components to be written
              x←(n↑m)⎕FREPLACE tie,c[p←3≠¯1↑i]
              x←(n↓m)⎕FREPLACE tie,c[~p]
              m←q←0                                          ⍝ reduce space consumption before recursion
              ~p:(1↑⍺),⍵ ∇ 1↓⍺                               ⍝ [()] or ([)]
              (1↑⍵),⍺ ∇ 1↓⍵                                  ⍝ ([]) or [(])
          }
     
          ms←{                                               ⍝ merge sort components ⍵
              0∊⍴⍵:⍵                                         ⍝ 0 components
              1=≢⍵:⍵⊣(⍺⍺ ⎕FREAD tie,⍵)⎕FREPLACE tie,⍵        ⍝ 1 component: sort its contents
              (⌊0.5×≢⍵)((∇↑)merge(∇↓))⍵                      ⍝ merge sorted halves
          }
     
          ⍺⍺ ms cpts
      }

    ∇ test
      ⎕←'Test #1: show all files in \tmp with NO extension'
      ⍴⎕←''filesIn'\tmp'
      ⎕←'Test #2: show all files in \tmp with one of the default extensions'
      ⍴⎕←filesIn'\tmp'
      ⎕←'Test #3: show all files in \tmp with extension dws'
      ⍴⎕←'dws'filesIn'\tmp'
      ⎕←'Test #4: show all files in \tmp with extension txt or dws'
      ⍴⎕←'txt' 'dws'filesIn'\tmp'
      ⎕←'Test #5: show all files in \tmp with extension txt or abc (string arg)'
      ⍴⎕←'txt dws'filesIn'\tmp'
      ⎕←'Test #6: show all files in \tmp with extension t?t'
      ⍴⎕←'t?t'filesIn'\tmp'
      ⎕←'Test #7: show all files in \tmp with extension txt'
      ⍴⎕←'t[xyz]t'filesIn'\tmp'
    ∇

:EndNamespace ⍝ fileUtils  $Revision: 1183 $
