﻿:Namespace miscUtils
⍝ This namespace contains various small functions that can be used in
⍝ every day programming. They can also be used as code examples. 
⍝ Version 1.01 modified deComment to decomment regular namespaces.


⍝ ** Day/Time

⍝ Examples of simple date-to-days and days-to-date functions

    ∇ ndays←daysIn ts;n;⎕IO;M;r
    ⍝ Number of days since 1900. Good until 2100.
      ⎕IO←1 ⋄ r←⍴ts ⋄ ts←1900 100 100⊤,ts
      ndays←365×n←ts[1;]
    ⍝ Account for all leap years
      ndays←ndays+⌊n÷4
      M←0 31 59 90 120 151 181 212 243 273 304 334
      ndays←ndays+M[12⌊1⌈ts[2;]]-(0=4|n⌈1)∧3>ts[2;]
      ndays←r⍴ndays+ts[3;]
    ∇

    ∇ ts←dateOf d;b;M;n;⎕IO
    ⍝ Date from number of days (see fn 'daysIn').
      ⎕IO←1
      d←(×ts)+d-⌈n×ts←⌊d÷n←365.25 ⍝ years and remaining no of days
      M←0 31 59 90 120 151 181 212 243 273 304 334
      ts←(100×ts+1900)+n←+⌿M∘.<d-(59<d)∧b←0=4|ts⌈1
      ts←(100×ts)+d-M[n]+b∧d>60
    ∇

    ⍝ With these 2 functions we can write the following

    ∇ ts←ndays addDays date
    ⍝ Add n days to date
      ts←dateOf ndays+daysIn date
    ∇

    ∇ dst←dsTime ts;dl;dh;ds2;ds1;nd;t
    ⍝ Returns 1 if argument is in daylight savings time
    ⍝ Based on last saturdays before the end of april and october
      ds1 ds2←430 1031 ⍝ change to suit needs
    ⍝ ts is in ⎕ts form
      dl←daysIn ds1+t←10000×1⍴ts ⋄ dh←daysIn ds2+t ⍝ last days of apr+oct
    ⍝ Last sunday will be (for penultimate subtract another 7)
      dl←dl-7|dl ⋄ dh←dh-7|dh ⍝ note daysIn assumes sunday=day 0
      dst←(dl≤nd)∧dh>nd←daysIn 100⊥3⍴ts
    ∇

    ∇ nts←nsec addSecs ts;n;⎕IO;sh
    ⍝ Add 'n' seconds to timestamp in ⎕ts form
      ⎕IO←1
      n←-⍴sh←⍴ts
      ts←⍉(¯2↑1,⍴ts)⍴ts←(sh⌈n↑7)↑ts ⍝ adjust shape and rank
      n←0 86400⊤nsec+60⊥ts[4 5 6;]  ⍝ for each line of ts
      nts←n[1;]addDays 100⊥ts[⍳3;]
      nts←⍉(0 100 100⊤nts)⍪(0 60 60⊤n[2;])⍪ts[7;]
      nts←,⍣(1∊⍴sh)+nts ⍝ return to vector if it was a vector
    ∇
    ⍝ Ex: 1e9 addSecs 1956 5 6 ⍝ 1 billion seconds after 1956/5/6

    ∇ z←formatTs a;days;mths
    ⍝ Format a date in timestamp format:  hh:mm:ss  day dd mmm yyyy
      days←7 3⍴'TueWedThuFriSatSunMon'
      z←days[,⎕IO+7|1++/⌊0 0.41 0+1.25 30.6 1×1900 12 0|(3⍴a)-(a[1+⎕IO]≤2),3 0;] ⍝ day of week
    ⍝ Tack on time and date
      mths←12 3⍴'JanFebMarAprMayJunJulAugSepOctNovDec'
      z←(1 3⍴3↓a)z(a[2+⎕IO])(mths[,a[1+⎕IO]-~⎕IO;])(1⍴a)
      z←,'I2,2P<:>ZI3,X2,3A1,I3,X1,3A1,I5'⎕FMT z
    ∇

    ∇ str←Now
    ⍝ Format actual date/time
      str←formatTs ⎕TS
    ∇

    ∇ z←dayOfWeek a;days
    ⍝ Calculate day of week from a ⎕ts-style timestamp.
      days←'Tuesday' 'Wednesday' 'Thursday' 'Friday' 'Saturday' 'Sunday' 'Monday'
      z←⊃days[⎕IO+7|1++/⌊0 0.41 0+1.25 30.6 1×1900 12 0|(3⍴a)-(a[1+⎕IO]≤2),3 0]
    ∇

⍝ The following two functions are reverse of each other

    ⍝ To turn a date into a single dating back from 0000/3/1 (=day 1)
    ⍝ Years not >100 are considered to be current century.
    ⍝ dno is the relative day number in a pseudo-julian calendar
    ⍝ The actual astronomers' julian day numbers may be
    ⍝ calculated by adding 1721119 to the result.

    ∇ dno←toJul date;yr;mo;day
    ⍝ Converts gregorian dates to pseudo-julian day numbers
    ⍝ date must be vector or matrix of 3 columns in year/month/day format.
      yr←+/1 0 0/date ⋄ mo←+/0 1 0/date ⋄ day←+/0 0 1/date
      yr←0 100⊤yr-mo≤2 ⍝ split at century
      dno←(⌊36524.25×yr[⎕IO;])+⌊365.25×yr[1+⎕IO;]
      dno+←day+⌊0.41+(30.6×12|mo-3)
    ∇

    ⍝ To turn the same number back into a date use this function:
    ∇ date←toGreg dayno;mo;day;century;year
    ⍝ dno is an array of numbers representing a julian day to be
    ⍝ converted to a gregorian date in (year month day) format
      century←⌊(dayno-0.25)÷36524.25 ⋄ day←⌊¯0.25+dayno-century×36524.25
      year←⌊(day+0.75)÷365.25 ⋄ day←⌊1.75+day-365.25×year ⋄ mo←⌊(day-0.59)÷30.6
      date←((century×100)+year+mo≥10),(1+12|mo+2),[(⍴⍴dayno)+⎕IO-0.5]⌊0.41+day-mo×30.6
    ∇

⍝ The following 2 functions can be used to transform RDCI values

    ∇ t←tsToRdci ts;s;⎕IO
    ⍝ Convert ⎕TS style data into ⎕FRDCI timestamp form
    ⍝ Accepts a numeric vector or array in ⎕TS format with 7≥¯1↑⍴argument
    ⍝ Returns time and date in ⎕FRDCI format with shape ¯1↓⍴argument
      ⎕IO←0 ⋄ s←¯1↓⍴ts ⋄ ts←((1⌈×/s),¯1↑⍴ts)⍴ts
      ts←((1↑⍴ts),7)↑ts ⍝ years<100 are in window 1950-2049
      ts[;0]+←(ts[;0]≤99)×(50/2000 1900)[100|ts[;0]]
      ts[;6]×←6÷100     ⍝ turn milisecs into 1/60th units
      t←¯60 ¯29 ¯1 30 60 91 121 152 183 213 244 274[ts[;1]-1]
      t-←(4|ts[;0])<2≥ts[;1]
      t←(t+⌊365.25×¯60+1900|ts[;0]),0 2↓ts
      t←¯18626112000+s⍴1 1 24 60 60 60⊥⍉t
    ∇

    ∇ ts←rdciToTs ts;md;s;sm;yr;z;⎕IO
    ⍝ Convert ⎕FRDCI timestamps to ⎕TS form
    ⍝ Returns ⎕TS style timestamps (shape (⍴argument),7)
      ⎕IO←0 ⋄ s←⍴ts ⋄ ts←,ts+18626112000
      md←365.2501|1+1461|yr←⌊ts÷5184000
      sm←31 61 92 122 153 184 214 245 275 306 337 366
      z←(,⍉<⍀sm∘.≥md)/,((⍴md),12)⍴⍳12
      md←(1+12|z+2),[0.1]⌈md-(0,sm)[z]
      ts←(1960+⌊(yr+60)÷365.25),md,⍉24 60 60 60⊤ts
      ts[;6]←⌊0.5+ts[;6]×100÷6
      ts←(s,7)⍴ts
    ∇

⍝ ** Math and accounting

    ∇ (n a s)←olds addStats obs;so2
    ⍝ Add new observations to old statistics
    ⍝ olds: number of observations, average, std deviation (0 if n=0)
      (n a s)←olds
      so2←(obs+.×obs)+n×(a×a)+s×s ⍝ sum obs*2
      n←n+⍴,obs           ⍝ new number of observations
      a←a+(obs+.-a)÷n     ⍝ new average
      s←((so2÷n)-a×a)*0.5 ⍝ new standard deviation
    ∇
    ⍝ Ex: (0 addStats 4 6 8)≡(0 addStats 4 6)addStats 8

    ∇ v←sum adjustSum v;real;dif;∆;i;h;s
    ⍝ Adjust integers v to add up to sum (accounting)
      real←v×sum÷+/v
      →(0=∆←sum-+/v←⌊0.5+real)⍴0 ⍝ exit if they all add up
      :Repeat ⍝ adjust
          dif←v-real ⋄ s←×∆ ⋄ i←i/⍳⍴i←s=-×dif ⋄ dif←|dif[i]
          h←dif⍳⌈/dif ⋄ v[i[h]]+←s ⋄ ∆←∆-s
      :Until 0∊|∆
    ∇
    ⍝ ex: 4 3 3 ≡ 10 adjustSum 3 3 3

    ∇ a←Average v
    ⍝ average of a list of numbers
      a←(+/v)÷⍴,v
    ∇

    ∇ r←Correlation a;nc;sa
    ⍝ Simple bivariate correlation of 2 variable given by the rows of A
      ⎕SIGNAL((2=⍴⍴a)∧2=1↑⍴a)↓11
      ⎕SIGNAL(nc←1↓⍴a←(∧⌿a≠0)/a)↓11
      r←''⍴÷nc ⋄ sa←+/a
      r←((+/×⌿a)-r××/sa)÷(×/(+/a*2)-r×sa*2)*0.5
    ∇

    ⍝ Prime numbers up to ~1000. The list could be expanded
    PRIMES←1 2 3 5 7 11 13 17 19 23 29 31 37 41 43 47 53 59 61 67 71 73 79 83 89 97 101 103 107 109
    PRIMES,←113 127 131 137 139 149 151 157 163 167 173 179 181 191 193 197 199 211 223 227 229 233
    PRIMES,←239 241 251 257 263 269 271 277 281 283 293 307 311 313 317 331 337 347 349 353 359 367
    PRIMES,←373 379 383 389 397 401 409 419 421 431 433 439 443 449 457 461 463 467 479 487 491 499
    PRIMES,←503 509 521 523 541 547 557 563 569 571 577 587 593 599 601 607 613 617 619 631 641 643
    PRIMES,←647 653 659 661 673 677 683 691 701 709 719 727 733 739 743 751 757 761 769 773 787 797
    PRIMES,←809 811 821 823 827 829 839 853 857 859 863 877 881 883 887 907 911 919 929 937 941 947
    PRIMES,←953 967 971 977 983 991 997 1009 1013 1019 1021 1031 1033 1039 1049 1051 1061 1063 1069

    ∇ factors←factorsOf n;⎕IO;p;test;⎕CT
    ⍝ Prime factors of n
      ⎕CT←0 ⋄ ⎕IO←1 ⋄ factors←n
      →(n∊PRIMES)/0
      test←1↓(PRIMES≤⌊n*0.5)/PRIMES
      :If 0=⍴factors←(0=test|n)/test   ⍝ any prime factors?
          ⎕SIGNAL(n>(¯1↑PRIMES)*2)⍴16  ⍝ beyond our limit?
          factors←n
      :Else
    ⍝ We now know which numbers divide n, there may be multiple instances:
    ⍝ ok, tack on the rest
          :If 1<n←n÷×/factors
              factors,←factorsOf n
              factors←factors[⍋factors]
          :EndIf
      :EndIf
    ∇

    ∇ distribution←classes freqDistr data;i
    ⍝ Returns the frequency distribution of the data in the classes.
    ⍝ 'classes' is a vector of lower class limits in ascending order.
    ⍝ e.g. ¯1.5 5 75 freqdist ¯3 2 99 4 ¯1.5  ←→  3 0 1.  ⎕io-independent.
      i←(((⍋classes,data)<⎕IO+⍴,classes),1)/⍳1+(⍴,classes)+⍴,data
      distribution←¯1+(1↓i)-¯1↓i
    ∇

    ∇ r←ratioOf x
    ⍝ Rational form of x
      r←(x∘.*1 0)÷x∘.∨1 1
    ∇

    ∇ r←roundInt x
    ⍝ Round x to the nearest even integer
      r←⌊x+0.5×0 0.5∨.≠2 1⊤x
    ∇

⍝ ** Programming utilities

    ∇ {msg}assert ok
    ⍝ Test assertion, ⎕signalling with <msg> if failure.
      :If 0=⎕NC'msg' ⋄ msg←'assertion failed' ⋄ :EndIf ⍝ default msg
      msg ⎕SIGNAL ok↓999
    ∇

    ∇ ⍙r←{⍙n}benchMark eXp;fF
    ⍝ Benchmark expressions
    ⍝ eXp is a list of expressions (or TABLE)
    ⍝ To avoid output assignment to local _ can be used
      '⍙n'default 100
      (-⎕IO-⍳1+⍴eXp)⎕MONITOR ⎕FX(⊂'fF;_'),eXp←,↓⍣(1=≡1/eXp)+eXp
      ⎕CS ⎕IO⊃⎕NSI
      :Repeat ⋄ fF ⋄ :Until 0≥⍙n←⍙n-1
      ⍙r←1 ¯1↓⎕MONITOR'fF'
      ⍙r←eXp,(0 0 1 1/⍙r)÷1⌈⍙r[;2/⎕IO] ⍝ line, average CPU, elapsed
    ∇

    ∇ deCommentWs
      deComment #
    ∇

    ∇ deComment where_;Obj
    ⍝ Remove comments in all functions in a namespace
      :For Obj :In where_.⎕NL-3 4 9.1 9.4 9.5
          :If 9≠where_.⎕NC Obj
              where_ deCommentFn Obj  ⍝ fn/op
          :Else
              :If ⎕THIS≠where_⍎Obj
                  where_ deCommentNs Obj  ⍝ ns/class/interface
              :EndIf
          :EndIf
      :EndFor
    ∇

    ∇ wh_ deCommentFn f__;t_1;t_2
    ⍝ Decomment function f__. Keep the line numbering.
    ⍝ wh_ is the namespace where it is found.
      :If 0∊⍴t_1←wh_.⎕CR f__
      :OrIf ~(10|⎕DR t_1)∊0 2
          t_2←↓∧\('⍝'≠t_1)∨≠\''''=t_1
      :OrIf ~(10|⎕DR wh_.⎕FX t_2/¨↓t_1)∊0 2
          ⎕←f__,' is impossible to decomment'
      :EndIf
    ∇

    ∇ wh_ deCommentNs f__;t_1;t_2;ok
    ⍝ Decomment ns f__. Keep the line numbering.
    ⍝ wh_ is the namespace where it is found.
      :Trap ok←0
          t_1←wh_.⎕SRC wh_⍎f__ ⋄ ok←1
          t_2←{∧\('⍝'≠⍵)∨≠\''''=⍵}¨t_1
          wh_.⎕FIX t_2/¨t_1
      :Else
          :If ok
              ⎕←f__,' cannot be decommented'
          :Else ⍝ try a regular ns
              deComment wh_⍎f__
          :EndIf
      :EndTrap
    ∇
    ⍝ Ex.: # deCommentFn 'myFn' ⍝ remove comments in function <#.myFn>

    ∇ recommentWs pos_;all_fns;where_
    ⍝ Reposition comments in all functions in a workspace/namespace
    ⍝ pos_ is no of leading spaces in front of comments,
    ⍝ it may be followed by the namespace to use.
      :If 326∊⎕DR pos_ ⋄ (pos_ where_)←pos_
      :Else ⋄ where_←⍎⎕IO⊃⎕NSI
      :EndIf
      all_fns←where_.⎕NL 3
      :While 0<1↑⍴all_fns
          where_ recommentFn(all_fns[⎕IO;])pos_
          all_fns←1 0↓all_fns
      :EndWhile
    ∇

    ∇ ns_ recommentFn(f__ pos_);t_1;t_2
    ⍝ Recomment f__ at pos_ in namespace ns_
    ⍝ Lines starting with a ⍝ will be left justified
    ⍝ Other lines with a comment will have the ⍝ at position pos_ if possible
      :If 0∊⍴t_1←ns_.⎕CR f__
     ⍝ Is this feasible? Some representations cannot be modified.
      :OrIf ~(10|⎕DR t_1)∊0 2
         ⍝ We could introduce a syntax to force comments to remain where they are
         ⍝ or tags to be considered as code but this version does not do that.
          t_2←↓∧\('⍝'≠t_1)∨≠\''''=t_1 ⍝ find the comments' mask
         ⍝ Labels are not taken into account
          t_1←(↓t_1){ ⍝ process each line 1 by 1
              com←'⍝ ',{(+/∧\' '=⍵)↓⍵}1↓(~⍵)/⍺ ⍝ left justify comments
              ' '∧.=code←⍵/⍺:com               ⍝ 1 line comments remain like this
              keep←pos_⌈1++/∨\⌽' '≠code        ⍝ how many characters to keep
              (keep↑code),com
          }¨t_2
      :OrIf ~(10|⎕DR ns_.⎕FX t_1)∊0 2
          f__,' is impossible to recomment'
      :EndIf
    ∇

    ⍝ Fix overflowed integers
    ∇ r←fixInt int
    ⍝ Bring numbers back into the 32b world
      r←⌊1 ¯2147483648⊥2 2147483648⊤int
    ∇

    ⍝ Hexadecimal and other base number handling
    ⍝ Arithmetic. We could write an <add> function, a <subtract> fn, etc.
    ⍝ instead we write an operator that will take the base and the function
    ⍝ to apply and define all other application hex function from it.

    ∇ result←la(fn inBase base)ra;basestr;number
    ⍝ Maximum base 36. Integers only.
      basestr←base↑⎕D,⎕A ⍝ digits, capital letters
      number←(basestr convertDecimal la)fn(basestr convertDecimal ra)
      result←basestr convertBase number
    ∇

    ∇ nb←base convertDecimal str;⎕IO;s;⎕ML
    ⍝ Convert a character array representing numbers in base
    ⍝ to decimal (ignore if already decimal).
      ⎕ML ⎕IO←0
      :If ' '∊1↑0⍴nb←str ⍝ only if character
          'number not in base'⎕SIGNAL(∧/str∊base,' ')↓999
         ⍝ Multiple spaces will create empty strings
          s←1↓¨(s=' ')⊂s←' ',str
          nb←(⍴base)⊥¨base∘⍳¨s~⊂''
      :EndIf
    ∇

    ∇ str←basestr convertBase number;⎕IO
    ⍝ Convert one or more decimal numbers to base (ignore if already character).
      :If 0∊1↑0⍴str←number ⍝ only for numbers
          ⎕IO←0
         ⍝ To ensure negative number get represented properly we use
         ⍝ their 2's complement on 32 bits
          str←,' ',⍉basestr[(⍬⍴⍴basestr)∘⊥⍣¯1{⍵+(2*32)×⍵<0}number]
      :EndIf
    ∇

    ⍝ Add two hexadecimal numbers, producing a hexadecimal sum.
    addHex ← + inBase 16
    ⍝ Subtract hexadecimal numbers, producing a hexadecimal result.
    subHex ← - inBase 16
    ⍝ Add two octal numbers, producing an octal sum.
    addOct ← + inBase 8
    ⍝ Subtract two octal numbers, producing an octal result.
    subOct ← - inBase 8
    ⍝ Because the functions accept numeric (decimal) arguments we can do
    ⍝ ' 14' ≡ 10 addHex 'A'


⍝ ** General utilities

    ⍝ This utility should be :Included or copied in the
    ⍝ namespace where it is to be executed.

    ∇ ∆00←∆01 Default ∆02;∆Ex;∆Rs
    ⍝ Default var ∆01 by value ∆02. '←' in name returns
    ⍝ the value instead and '⍎' will use ⍎ of ∆02 instead of ∆02 itself.
      ∆Rs←'←'∊∆01 ⋄ ∆Ex←'⍎'∊∆01
      ⎕CS ⎕IO⊃⎕NSI ⍝ execute in original namespace
      ∆Ex←⎕IO+2⌊∆Ex+⎕NC ∆01←∆01~'⍎←'
      ⍎((∆Rs+⎕IO)⊃∆01'∆00'),'←',,∆Ex⊃'∆02'∆02 ∆01
    ∇
    ⍝ 'abc' default 666 ⍝ if 'abc' does not exist it is assigned the value 666
    ⍝ '⍎XYZ' default '⎕FREAD Tie,1' ⍝ if XYZ does not exist it is assigned cpt 1
    ⍝ '←OK' default 1 ⍝ if OK is undefined the value 1 is RETURNED (OK not set)
                      ⍝ otherwise the value of OK is returned
    ⍝ ⍎ and ← can be combined

    ∇ r←a catMat1 b
    ⍝ Catenates 2 maximum rank 2 objets of different shapes on 1st axis
    ⍝ Empty vectors are replaced by 0 0⍴''
      a←(¯2↑1,(×⍴,a),⍴a)⍴a
      b←(¯2↑1,(×⍴,b),⍴b)⍴b
      r←0,1↓(⍴a)⌈⍴b
      a←(r⌈⍴a)↑a ⋄ b←(r⌈⍴b)↑b
      r←a⍪b
    ∇

    ∇ index←left charmatIndex right;grade;rows;select;iota;i
    ⍝ Returns the locations of the rows of right within left for character matrices
    ⍝ left and right need not match in length.
    ⍝ Much more efficient than ∧.= for large matrices
      i←0,1↓(⍴left)⌈⍴right←(¯2↑1 1,⍴right)⍴right
      left←(i⌈⍴left)↑left ⋄ right←(i⌈⍴right)↑right
      left←left⍪⎕AV[⎕IO+255]
      grade←⎕AV⍋right⍪left
      select←grade<⎕IO+rows←1↑⍴right
      index←iota←⍳rows
      index[select/grade]←((~select)/grade)[⎕IO+(select/⍳⍴select)-iota]-rows
      index[(∨/right≠left[index;])/iota]←⎕IO+(1↑⍴left)-1
    ∇

    ∇ result←origins fromTo ends;s;t
    ⍝ Generate all the numbers from a number up to another
    ⍝ A list can be supplied
    ⍝ e.g.  1 157 ¯12 fromto 2 161 ¯10 ←→ 1 2 157 158 159 160 161 ¯12 ¯11 ¯10.  ⎕io-independent.
      t←+\⎕IO,1+ends-origins
      s←((¯1↑t)-⎕IO)⍴1
      s[¯1↓t]←origins-¯1↓0,ends
      result←+\s
    ∇

    ⍝ This <If> function can be used in places like
    ⍝ ⎕SIGNAL 11 If Moon=blue
    If←/⍨

    ∇ result←indexGen vector;s;v
    ⍝ Generate ⊃,/⍳¨vector without ¨ but works if vector is empty.
    ⍝ result←(⍳1↑vector),(⍳1↑1↓vector),(⍳1↑2↓vector),...,⍳¯1↑vector.  ⎕io-responsive.
      v←(vector≠0)/vector
      s←(+/v)⍴1
      s[⎕IO++\¯1↓v]←1-¯1↓v
      result←(+\s)-~⎕IO
    ∇

    ∇ indices←{sequence}proIndexOf vector
    ⍝ Progressive dyadic iota
    ⍝ e.g.  [⎕io←1] 'aba' proIndexOf 'acaaba' ←→ 1 4 3 4 2 4.
      sequence←,sequence
      indices←((⍋sequence⍳sequence,vector)⍳⍳⍴sequence)⍳(⍋sequence⍳vector,sequence)⍳⍳×/⍴vector
    ∇

⍝ ** Curiosities

    ⍝ BlackJack Partionning: partition a list into a maximum size
    ∇ r←size bjPart list;a;b
      r←1=(a↑1)⌹((2⍴a)⍴(1+a←⍴,list)↑1)-<⍀b∘.>size+¯1↓0,b←+\list
    ∇
    ⍝ caveat emptor: this is not a fast function for large list

    ⍝ Find where text delimited by ' or " starts/ends
    ∇ mask←Quotes str;Q;qm;tq;n
      qm←str∊'''"' ⋄ n←⍴Q←qm/str   ⍝ work on quotes only
      tq←<\(∘.<⍨⍳n)∧∘.=⍨Q          ⍝ matching trailing markers
      mask←n⍴tq⌹(∘.=⍨⍳n)-0 ¯1↓0,tq ⍝ all trailing markers
      mask←qm\mask∨¯1↓1,mask       ⍝ lead & trail markers
    ∇
    ⍝ ex:
    ⍝ 0 1 0 0 0 1 0 0 1 0 1 0 ≡ Quotes ⍞
    ⍝["I'm"--'"']

    ⍝ Easter date
    ∇ z←Easter y;a;c;d;e;g;m;p;s;t;u;v;x
    ⍝ y is the number of a Gregorian year;
    ⍝ z is the month and day on which Easter Sunday falls in that year.
    ⍝ For example, Easter 1989 ←→ 3 26. Easter falls on the first
    ⍝ Sunday following the first full moon which occurs on or after
    ⍝ March 21. See Augustus DeMorgan, "A Budget of Paradoxes", and
    ⍝ Donald Knuth, CACM 5, 4, April 1962, "The Calculation of Easter...",
    ⍝ and his "Fundamental Algorithms", Section 1.3.2, problem 14.
    ⍝
    ⍝ a is the Golden Number, the number of the year in the 19-year
    ⍝ Metonic cycle, used to determine the position of the Moon. (The
    ⍝ phases of the Moon have a cycle of approximately 19 years).
      a←1+19|y
    ⍝ c is the number of the century in which y falls. This algorithm
    ⍝ accepts the year xx00 as occuring in the xx+1th century.
      c←1+⌊0.01×y
    ⍝ g is the Gregorian correction, the number of preceding years
    ⍝ like 1700, 1800, 1900 when leap year was not held.
      g←¯12+⌊0.75×c
    ⍝ v is the clavian correction, a correction of the Metonic cycle
    ⍝ of about 8 days every 2500 years, used to synchronize Easter
    ⍝ with the Moon's orbit.
      v←¯5+⌊0.04×5+8×c
    ⍝ x is the extra days specifying when Sunday occurs in March.
    ⍝ (March 7|-x will be a Sunday).
      x←(¯10+⌊1.25×y)-g
    ⍝ p is the approximate age of the calendar moon at the beginning
    ⍝ of the year.
      p←30|20+v-g-11×a
    ⍝ e is the epact, a better approximation than p.
      e←p+(p=24)∨(p=25)∧a>11
    ⍝ t is the day of the full moon, calculated using s.
      t←s+30×21>s←44-e
    ⍝ u is the day of the following Sunday, but may be greater than 31.
      u←t+7-7|x+t
    ⍝ d is the day of Easter Sunday.
      d←u-31×u>31
    ⍝ m is the number of the month, 3 (March) if u was less than 32,
    ⍝ and 4 (April) otherwise.
      m←3+u>31
    ⍝ z is the month number and day number of Easter in year y.
      z←m,[⎕IO+(⍴⍴d)-0.1]d ⍝ return month day for each date
    ∇

    ∇ r←InternetConnected;IC
    ⍝ 1 if the Internet is there
      'IC'⎕NA'I WinINet|InternetGetConnectedState I I'
      r←IC 0 0
    ∇

:EndNamespace ⍝ miscUtils  $Revision: 739 $ 