﻿:Namespace textUtils
⍝ Namespace containing utilities to process character strings (text)
⍝ Because this namespace can be :included we need to localize the system
⍝ variables (⎕vars) in each function as well as the global constants like CR
⍝ use ⎕C
⍝ V1.06 2021-02-18
    
    ⎕io←1 ⋄ ⎕ml←2
    (LOWER UPPER)←{⎕io←0 ⋄ c[;⍳26]←⊖2 26↑c←(i←⎕av⍳⍵)⌽⊃2⍴⊂⎕av⋄ ⋄ ↓(-i)⌽c}'Aa'
    lowerCase←¯1∘⎕C
    upperCase←1∘⎕C
    (BS LF CR)←1↓4↑⎕av
    
    ∇ t←addQuotes t;q
    ⍝ Add quotes to 't' by enclosing it in quotes, and doubling all quotes in it.
      t←q,((1+t=q)/t),q←''''
    ∇
    
    ∇ text←squeezeText text;b
    ⍝ Remove double spaces from text
      text←(b⍲1⌽b←text∊' ')/text←' ',text
    ∇
    
    ∇ r←{width}centerText s
    ⍝ Center String on Width colums
    ⍝ S can be any structure; Width defaulted by ⎕pw
      :If 0=⎕NC'width' ⋄ width←⎕PW ⋄ :EndIf
      r←(-1⌈⍴⍴s)↑1,⍴s
      r←(r,-width)↑s,((r←¯1↓r),⌊(width-¯1↑r)÷2)⍴' '
    ∇
    ⍝ NOTE: blanks not taken into account
    
    ∇ text←text charReplace c1c2
    ⍝ Replace occurrences of char1 in text by char2
      ((text=1↑c1c2)/text)←1↓c1c2
    ∇
    
    ∇ newtext←{cut}intoCR text;lm;CR;LF
    ⍝ Transform text into CR delimited lines
      LF CR←2↓4↑⎕AV
      :If ∨/lm←CR LF⍷newtext←text
          newtext←(~¯1⌽lm)/text
      :ElseIf ∨/lm←LF=text
          (lm/newtext)←CR
      :EndIf
     
    ⍝ If lines need to be nested:
      :If 2=⎕NC'cut'
      :AndIf cut
          ⎕ML←1
          newtext←1↓¨(1,lm)⊂'.',newtext
      :EndIf
    ∇
    
    ∇ m←{d}intoMatrix v
     ⍝ Matrix representation from Vector
     ⍝ d is the delimiter, defaulted by CR
      :If 0≠⎕NC'd' ⋄ ((v∊d)/v)←⎕AV[3+⎕IO] ⋄ :EndIf
      m←⎕FMT v
    ∇
    
    ∇ lj←leftJustify strs
    ⍝ Left justify vector or matrix by shifting spaces before
      lj←{(+/∧\' '=⍵)⌽⍵}strs
    ∇
    
    ∇ rj←rightJustify strs
    ⍝ Right justify vector or matrix by shifting spaces after
      rj←{(-+/∧\' '=⌽⍵)⌽⍵}strs
    ∇
    
    ∇ te←trimEnds strs;⎕ML;⎕IO
    ⍝ Trim ends of vector or matrix by dropping spaces at the end
      ⎕ML←2 ⋄ ⎕IO←0 ⋄ te←⊃⌽∘{(0⍳⍨' '=⍵)↓⍵}⍣2¨↓strs
    ∇
    
    ∇ text←lines mergeWith chars;⎕ML
    ⍝ Turns a series of lines into a single string
    ⍝ with 'chars' between each line merged
      ⎕ML←1 ⋄ text←(⍴,chars)↓∊chars∘,¨lines
    ∇
    
    ∇ newtext←text splitAt offset;lm;⎕ML;⎕IO
    ⍝ Cut text at offset positions (0 included)
      ⎕ML←0 ⋄ ⎕IO←0
      lm←(⍴text)↑1 ⋄ lm[offset]←1
      newtext←lm⊂text
    ∇
    
    ∇ (stem leaf)←string splitLast char;take;drop;⎕IO
    ⍝ Split string into 2 pieces according to last char in it
      ⎕IO←1 ⋄ drop take←0 1-char⍳⍨⌽string
      (stem leaf)←(drop↓string)(take↑string)
    ∇
    
    ∇ newtext←text splitOn chars;lm;⎕ML
    ⍝ Cut text using chars as delimiters
      ⎕ML←1
      lm←text∊chars
      newtext←1↓¨(1,lm)⊂'.',text
    ∇
    
    ∇ pieces←{where}cutString string;⎕ML
    ⍝ Cut string using first character and excluding it
      :If 0=⎕NC'where' ⋄ where←string=1↑string ⋄ :EndIf
      ⎕ML←2 ⋄ pieces←1↓¨where⊂string
    ∇
    
    ⍝ The next pair of functions can be used to reformat text quickly.
    ⍝ For example, to make a variable containing text in left aligned paragraphs
    ⍝ fit a width of 70 do
    ⍝    new← 70 reshapeText condRavel text
    
    ∇ txt←condRavel txt;crs;s;CR;LF;ncr
    ⍝ Conditionally ravel text where CR are followed by non-white Spaces
    ⍝ e.g. 'x',CR,'x' is changed by 'x x'
      (LF CR)←2↓4↑⎕AV
      :If 1<⍴⍴txt ⋄ txt←1↓,CR,txt ⋄ :EndIf
      txt[(txt=LF)/⍳⍴txt]←CR ⍝ LF=CR
    ⍝ We first remove the spaces at the end of each line
      ncr←CR≠1↑txt
      txt←{(⍵∊CR)⊂⍵}(ncr↑CR),txt ⍝ each line
      s←-⊥⍨¨' '=¨txt ⍝ number of spaces to drop at the end of each line
      txt←⊃,/s↓¨txt  ⍝ drop them, merge lines
      crs←s/⍳⍴s←0,1↓¯1↓CR=txt ⍝ find where the CRs are and replace by
      txt[(⍱⌿txt[¯1 1∘.+crs]∊CR,' ')/crs]←' ' ⍝ space those between words
      txt←ncr↓txt    ⍝ remove CR added
    ∇
    
    ∇ rfmtxt←{n}reshapeText text;⎕IO;b;d;line;n1;s
    ⍝ Embed the CR character into the text so lines are
    ⍝ approximately and at most 'n' long. Existing CRs are left in.
    ⍝ If the text contains CRs to be removed use function <condRavel>.
      :If 0=⎕NC'n' ⋄ n←⎕PW ⋄ :EndIf
      ⎕IO←0 ⋄ rfmtxt←'' ⋄ text←,text ⋄ n1←1+n
      :While n<⍴text
          :If 0∊s←CR≠line←n1↑text            ⍝ any CR in line?
              rfmtxt←rfmtxt,(-d←s⊥s)↓line    ⍝ keep up to last CR
              text←(n1-d)↓text
          :ElseIf ¯1↑s←' '=line              ⍝ No CRs in line; does the line fit?
              rfmtxt←rfmtxt,((-s⊥s)↓line),CR ⍝ it does
              text←n1↓text
          :ElseIf ~∨/s                       ⍝ cut at last space. Any?
              rfmtxt←rfmtxt,(¯2↓line),'-',CR ⍝ no, add a dash too
              text←(n-1)↓text
          :Else ⍝ line contains spaces but the last word does not fit
              b←(b⍳1)↓b←⌽s ⋄ text←(⍴b)↓text  ⍝ forget last word
              d←(⍴b)-b⍳0                     ⍝ number of chars to take
              rfmtxt←rfmtxt,(d↑line),CR
          :EndIf
      :EndWhile
      rfmtxt←rfmtxt,text
    ∇
    
∇      r←{options}blockAlign words;sh;blk;sp;pw;⎕ML;⎕IO;n
     ⍝ Fit Words ⎕PW wide aligned on blocks of 4 cells, spaced 1 apart.
     ⍝ Options allow changing the width, the blocks' width and the spacing.
      ⍎(0=⎕NC'options')/'options←⍬' ⋄ (pw blk sp)←options,(⍴,options)↓⎕PW,4 1 ⋄ pw←pw+sp
      ⎕ML←⎕IO←1 ⋄ r←0⍴⊂'' ⋄ sh←∊⍴¨,¨words
      sh←∊⍴¨words←(blk×⌈blk÷⍨sh+sp)↑¨words ⍝ adjust each word's size
      :While 0<⍴words
          r←r,,/(n←+/pw≥+\sh)↑words ⋄ (words sh)←n↓¨words sh
      :EndWhile
      r←(0,-sp)↓↑r
∇
:EndNamespace ⍝ textUtils  $Revision: 1628 $ 
