﻿:Class Asymmetric

⍝ Asymmetric encryption uses a pair of keys to encrypt and decrypt. There
⍝ is a "public" key which is used to encrypt. Decrypting, on the other hand,
⍝ requires both the "public" key and an additional "private" key. The advantage is
⍝ that people can send you encrypted messages without being able to decrypt them.
⍝ The only provider supported is the RSACryptoServiceProvider.

    :include CryptTools  ⍝∇:Require =\crTools
    ⎕using← DNetCrypto

⍝ WARNING:
⍝ Underlying base, enhanced, or strong cryptographic service provider (CSP)
⍝ implementations create a key container for storing an RSA public/private
⍝ asymmetric key pair. Key containers are stored in user profiles.
⍝ For performance reasons, the user profile is not loaded by the system under
⍝ a scenario where the .NET code runs in a Web service, ASP page, or COM+.
⍝ If the user profile is not loaded, a key container cannot be opened or
⍝ created. Because a key container is required for RSACryptoServiceProvider
⍝ or DSACryptoServiceProvider, the .NET code (when run from a Web Service,
⍝ ASP Page, or COM+) fails by throwing the exception "CSP for this implementation could not be acquired"
⍝ If the same .NET code is run from the context of the interactive logged-on
⍝ user, where the user profile is loaded by Winlogon, a key container can be
⍝ created and opened, and the code works under this scenario.

    :field _KeyContainerName ←'Encryption.AsymmetricEncryption.DefaultContainerName'
    :field _UseMachineKeystore←1
    :field _Ksize ←1024
    :field public PublicKEY
    :field public PrivateKEY

⍝ This is an example of key generated by <GenerateNewXMLKeys> (sans <tags>)
    KeyExample← 'yiYdUT+Pf3+tRYsYl9IK3gYC0Iyc08ch6FFWvLPs3gCT2xUZsTq+EALJCOJ1fiEE9e1m0Lv3mvoKc3ioOJ2Ljzg+kbZYdzqfbAY+exeisb+OxJUcw3gnZizm9avXmO9oGtSJ2p2WaB/NpQSN+gRqySaTLrP+OD41vKTnAwI9M9E=' 'AQAB' '6B7TX8yFfU1fCJ/7e+qu3o1TiR6ow+ngppSjLpSJ+tMdRlSmcFLWG/qF0+xt5FuL5bYEuQi4JangEoH9V5k7pw==' '3vHzu2eDwCmQQDVwj2gPO9A9iWRIdT+rFOFGpsAURc9p3iL+i9o76L+ccIedz//58cmc4bkwvaWbeFJbUEgjxw==' 'FH1MGUO7cNEYl3FPyKNxpjGBczQin5CQhQPP17fC6kIibmfgfT/CqJuXnppbJPIzyiYkCpVLc/+nyNpym33qNw==' 'r7l1qxkEN8qK8JEuj1ot569TUZ/BnOj6KBMIZRaw4/WANJ7Qk9sHrYSaAfhi4zeYAoQvK5uUV2bZDBWd88FosQ==' 'iDsfDI5wmNZJiOYS5wFtVTMDe+KfobITdlKYUIo7dNeHS0PzDGf7YOuHaq1gdIYMu/zFG+w9pIoDBXUsStxBew==' 'Od1/NurC4PRjXSzb9Vmi0rtVwuz9V4IBKsdd1NaMT2BK8sOvfxQZo+uoMGrixyAck2lJrSahUjORGkzHmq0+bVagjQar/vAgztRJZo1XtZ7r3prb/Vsav5UJjyHwjGHLHOOh0shJpuRJSXxD7/vq4pGYPyOiJ/UVRD16allDQCE='

   ⍝ Typical use:
   ⍝ A←⎕new Asymmetric 1024
   ⍝ A.GenerateNewKeys
   ⍝ E←A.Encrypt text

    ⎕io ⎕ml←1 ⋄ ⎕wx←3

    :Class PublicKey
        ⍝ Represents a public encryption key. Intended to be shared, it
        ⍝ contains only the Modulus and Exponent.

        ∇ make arg
      ⍝ This is where you make the public Key.
      ⍝ It can be made of INI files, XML, etc.
      ⍝ Here we only accept strings for the moment.
          :Implements constructor
          :Access public
          :If 1≡≡arg ⋄ makefromXML arg
          :Else
              (MOD EXP)←2↑arg ⍝ 2 strings
          :EndIf
        ∇

        ∇ makefromXML array;mat
          :Access public
          mat←⎕XML array
          (MOD EXP)←mat[mat[;2]⍳'Modulus' 'Exponent';3]
        ∇

        ∇ makefromINI file
          :Access public
          ÷⎕←'Not available'
        ∇
        ∇ r←makeXML
          :Access public
          r←⎕XML 0,'Modulus' 'Exponent',⍪MOD EXP
        ∇
        ∇ r←toParm;⎕USING
          :Access public
          ⎕USING,⍨←⊂'System'
          r←⎕NEW RSAParameters
          r.(Modulus Exponent)←Convert.FromBase64String¨MOD EXP
        ∇
    :endclass

    :Class PrivateKey
  ⍝ Represents a private encryption key. Not intended to be shared, as it
  ⍝ contains all the elements that make up the key.

        ∇ make arg
            ⍝ This is where you make the public Key.
            ⍝ It can be made of INI files, XML, etc.
            ⍝ Here we only accept XML strings for the moment.
          :Implements constructor
          :Access public
          :If 1≡≡arg ⋄ makefromXML arg
          :Else ⋄ (MOD EXP P Q DP DQ IQ D)←arg
          :EndIf
        ∇

        ∇ makefromXML array;mat;elem
          :Access public
          mat←⎕XML array
          elem←,¨'Modulus' 'Exponent' 'P' 'Q' 'DP' 'DQ' 'InverseQ' 'D'
          (MOD EXP P Q DP DQ IQ D)←mat[mat[;2]⍳elem;3]
        ∇
        ∇ r←makeXML
          :Access public
          r←⎕XML 0,'Modulus' 'Exponent' 'P' 'Q' 'DP' 'DQ' 'InverseQ' 'D',⍪MOD EXP P Q DP DQ IQ D
        ∇

        ∇ makefromINI file
          :Access public
          ÷⎕←'Not available'
        ∇

        ∇ r←toParm;all;⎕USING
          :Access public
          ⎕USING,←⊂'System'
          r←RSAParameters
          all←Convert.FromBase64String¨⊂¨MOD EXP P Q DP DQ IQ D
          r.(Modulus Exponent P Q DP DQ InverseQ D)←all
        ∇
    :endclass


    ∇ make1 Ksize
  ⍝ This is where you make the public Key.
  ⍝ It can be made of INI files, XML, etc.
  ⍝ Here we only accept [XML] strings for the moment.
      :Implements constructor
      :Access public
      :If Ksize≠0
          _Ksize←Ksize
          newrsa 0
      :Else ⍝ use Example key
          newrsa KeyExample
      :EndIf
    ∇

    ∇ make0         ⍝ use default 1024
      :Implements constructor
      :Access public
      newrsa 0
    ∇

    ∇ newrsa key
      _rsa←⎕NEW RSACryptoServiceProvider _Ksize
      :If key≡0 ⋄ GenerateNewKeys
      :Else ⋄ UseThisKey key
      :EndIf
    ∇

    ∇ GenerateNewKeys
    ⍝ Generate new keys for this instance
      :Access public
      (PublicKEY PrivateKEY)←GenerateNewKeysetFromXML GenerateNewXMLKeys
    ∇

    ∇ (publicKeyXML privateKeyXML)←GenerateNewXMLKeys;rsa
      :Access public
      rsa←RSACryptoServiceProvider.Create ⍬
      publicKeyXML←rsa.ToXmlString 0
      privateKeyXML←rsa.ToXmlString 1
    ∇

    ∇ UseThisKey Key
    ⍝ Generatekeys from the example supplied
      :Access public
      PublicKEY←⎕NEW PublicKey(2↑Key)
      PrivateKEY←⎕NEW PrivateKey Key
    ∇

    ∇ r←EncryptString d
      ⍝ Encrypt string
      :Access public
      r←RSAEncrypt(d(_rsa.ExportParameters 0)0)
    ∇

    ∇ encryptedData←RSAEncrypt(DataToEncrypt RSAKeyInfo DoOAEPPadding)
      :Access public
      _rsa.ImportParameters(RSAKeyInfo)
      encryptedData←_rsa.Encrypt(UCSN DataToEncrypt)DoOAEPPadding
    ∇

    ∇ r←DecryptString d
      ⍝ Encrypt string
      :Access public
      r←UTF8 RSADecrypt(d(_rsa.ExportParameters 1)0)
    ∇

    ∇ decryptedData←RSADecrypt(DataToDecrypt RSAKeyInfo DoOAEPPadding)
      :Access public
      _rsa.ImportParameters(RSAKeyInfo)
      decryptedData←_rsa.Decrypt DataToDecrypt DoOAEPPadding
    ∇

   ⍝ Generates a new public/private key pair as objects
    ∇ (publicKey privateKey)←GenerateNewKeysetFromXML arg
      :Access public
      :If arg≡1 ⋄ (publicKey privateKey)←GenerateNewXMLKeys
      :Else ⋄ (publicKey privateKey)←arg
      :EndIf
      publicKey←⎕NEW PublicKey(publicKey)
      privateKey←⎕NEW PrivateKey(privateKey)
    ∇

    ∇ r←EncryptPrivate d
      :Access public
      :Trap 0
          r←_rsa.Encrypt((UCSN d)0)
      :Else
          'data toolarge'⎕SIGNAL 99 if 1∊'bad length'⍷⎕EXCEPTION.Message
          ⎕SIGNAL 11
      :EndTrap
    ∇

    ∇ rsa←GetRSAProvider;csp
     
      :Trap 0
          csp←⎕NEW CspParameters
          csp.KeyContainerName←_KeyContainerName
          rsa←⎕NEW RSACryptoServiceProvider(_KeySize,csp)
          rsa.PersistKeyInCsp←0
          RSACryptoServiceProvider.UseMachineKeyStore←1
      :Else
        ⍝
      :EndTrap
    ∇

:EndClass ⍝ Asymmetric  $Revision: 739 $ 